%%%-------------------------------------------------------------------
%%% @author Michal Wrzeszcz
%%% @copyright (C) 2019 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% Module that allows traversing via different structures and execution of jobs on them.
%%% The basic concepts are tasks (see traverse_task.erl) and pools. Each task represents single traverse while pool
%%% represents single execution environment. The tasks can be scheduled to be executed on different environments
%%% (different clusters). Thus, concepts of task creator and executor are introduced. The creator represents the
%%% environment where the task is created while executor represents environment where the task is processed.
%%% The task is processed executing jobs. The job represents processing of single entity (e.g., dir or file).
%%% First job (called main_job) is provided to task start function, further jobs can be generated as first and next
%%% jobs result. First job is treated specially as it is the only job that exists before task start. Thus, its id is
%%% stored in a task document (see traverse_task.erl) and its special status is highlighted when it is persisted.
%%% Such special treatment is a result of the fact that task can be scheduled for other environment. Thus main job must
%%% be available there (all other jobs will be created on the environment that processes task).
%%% The jobs are divided into two groups: master and slave jobs. Master jobs process entities that define structure
%%% (e.g., directories) while slave jobs should process other elements (e.g., files). Thus, master jobs can produce
%%% further master jobs and/or slave jobs.
%%% Each pool uses two instances of worker pool for jobs execution: one instance for execution of master jobs and
%%% one for slave jobs. Each master job is blocked until all of slave jobs generated by it are finished.
%%% Worker pools can be started on several nodes. In such a case, load is balanced between nodes executing different
%%% tasks on different nodes (all jobs connected with single task are executed on the same node - see
%%% traverse_tasks_scheduler.erl).
%%% Job handling functions are provided by callback modules (see traverse_behaviour.erl). Callback modules provide also
%%% functions needed for jobs persistency (master jobs are stored only). They also can provide additional information
%%% for datastore documents synchronization and tasks sorting (see traverse_task_list.erl). Multiple callback modules
%%% can be used for single pool (different callback modules for different tasks).
%%% Task load balancing base on groups. Each task can be connected with a group and load balancing algorithm first
%%% chooses group and only than task to be executed (to prevent single group from taking all resources while other
%%% groups are waiting - see traverse_tasks_scheduler.erl).
%%%
%%% Typical task lifecycle is as follows:
%%%     - user calls run function
%%%     - the job that initializes task is persisted
%%%     - number of ongoing tasks is verified
%%%         - if the number is lower than the limit and task executor is local environment,
%%%             task will be created with status "ongoing" and number of parallel tasks is incremented
%%%         - otherwise, task will be created with status "scheduled"; if task executor is local environment group
%%%             connected with task is marked as waiting for execution
%%%     - if task is created as ongoing, it is put in master worker pool queue, next:
%%%         - workers from worker pool execute jobs connected with task (main job and jobs created as a result of master
%%%             jobs processing) until all jobs are finished
%%%         - last job connected with task updates task status to "finished"
%%%         - last job connected with task checks if there is any waiting tasks
%%%             - if some tasks are waiting, longest waiting group for execution is chosen, and oldest task from
%%%                 this group is put in master worker pool queue; chosen task status is changed to "ongoing"
%%%             - otherwise, number of parallel tasks is decremented
%%% Additionally, tasks may be started when task scheduled at other environment appears (and local environment is
%%% chosen for execution) or during environment restart. In first case, number of ongoing tasks is verified to check if
%%% task can be executed or has to wait.
%%% @end
%%%-------------------------------------------------------------------
-module(traverse).
-author("Michal Wrzeszcz").

-include("global_definitions.hrl").
-include("traverse/traverse.hrl").
-include("modules/datastore/datastore.hrl").
-include_lib("ctool/include/logging.hrl").

%% API
-export([init_pool/5, init_pool_service/5, restart_tasks/3, stop_pool/1, stop_pool_service/1,
    run/3, run/4, cancel/2, cancel/3, on_task_change/2, on_job_change/5]).
%% Functions executed on pools
-export([execute_master_job/10, execute_slave_job/5, is_job_cancelled/1]).
%% For rpc
-export([run_on_master_pool/10, run_task/3]).

%% @formatter:off
% Basic types for execution environment
-type pool() :: binary(). % term used to identify instance of execution environment
-type callback_module() :: module().
-type environment_id() :: traverse_task_list:tree(). % environment (cluster) where task is scheduled or processed
-type pool_options() :: #{
    executor => environment_id(),
    callback_modules := [callback_module()],
    restart => boolean(),
    % If the limit is not set, limit is equal to overall limit (argument of init_pool function) divided by nodes number
    parallel_orders_per_node_limit => traverse_tasks_scheduler:ongoing_tasks_limit()
}.
% Basic types for tasks management
-type id() :: datastore:key().
-type task() :: traverse_task:doc().
-type description() :: #{atom() => integer()}.  % map describing progress of task; framework provides following counters:
                                                % slave_jobs_delegated, master_jobs_delegated, slave_jobs_done,
                                                % master_jobs_done, slave_jobs_failed, master_jobs_failed;
                                                % the user can add own counters returning map with value upgrade from
                                                % job (see traverse_behaviour.erl)
-type additional_data() :: #{binary() => binary()}.
-type status() :: atom().   % framework uses statuses: scheduled, ongoing, finished and canceled but user can set
                            % any intermediary status using traverse_task:update_status function
-type group() :: binary(). % group used for load balancing (see traverse_tasks_scheduler.erl)
-type master_job_mode() :: single | all. % mode of master jobs starting on pool; for `all` each master job start is
                                         % triggered immediately after job definition; for `single` only one master
                                         % job start is triggered immediately after job definition - rest of jobs
                                         % waits until execution of triggered job is finished; as a result only one
                                         % master job is performed in parallel for each task for `single` mode
-type run_options() :: #{
    executor => environment_id(),
    creator => environment_id(),
    callback_module => callback_module(),
    group_id => group(),
    additional_data => additional_data(),
    master_job_mode => master_job_mode()
}.
-type task_execution_info() :: #task_execution_info{}. % see traverse.hrl
% Basic types for jobs management
-type job() :: term().
-type job_id() :: datastore:key().
-type job_status() :: waiting | on_pool | ended | failed | canceled.
-type master_job_map() :: #{
    slave_jobs => [job()],
    sequential_slave_jobs => [job()],
    master_jobs => [job()],
    async_master_jobs => [job()],
    description => description(),
    cancel_callback => job_cancel_callback(),
    finish_callback => job_finish_callback()
}.

-type master_job_extended_args() :: #{
    task_id := id(),
    master_job_starter_callback => master_job_starter_callback() % callback is not supported when master_job_mode =:= single
}.
-type master_job_starter_callback() :: fun((master_job_starter_args()) -> ok).
-type master_job_starter_args() :: #{
    jobs => [job()],
    cancel_callback => job_cancel_callback()
}.
-type job_cancel_callback() :: fun((description()) -> ok).
-type job_finish_callback() :: fun((master_job_extended_args(), description()) -> ok).
% Types used to provide additional information to framework
-type timestamp() :: non_neg_integer(). % Timestamp used to sort tasks (usually provided by callback function)
-type sync_info() ::  #{
    mutator => datastore_doc:mutator(),
    scope => datastore_doc:scope(),
    sync_enabled => boolean(),
    local_links_tree_id => datastore:tree_id(),
    remote_driver => datastore:remote_driver(),
    remote_driver_ctx => datastore:remote_driver_ctx()
}.
%% @formatter:on
% Internal types for framework
-type execution_pool() :: worker_pool:name(). % internal names of worker pools used by framework
-type ctx() :: traverse_task:ctx().
-type node_crash_policy() :: restart | cancel_task.
-type task_callback() :: task_started | task_finished | task_canceled | on_cancel_init.
% Types used by functions that restart tasks after node restart
-type tasks_ctxs() :: #{id() => ctx() | ctx_not_found}.
-type jobs_per_task() :: #{id() => [{job_id(), job()}]}.

-export_type([pool/0, id/0, task/0, group/0, master_job_mode/0, task_execution_info/0, job/0, job_id/0, job_status/0,
    environment_id/0, description/0, status/0, additional_data/0, master_job_extended_args/0, timestamp/0,
    sync_info/0, master_job_map/0, callback_module/0, node_crash_policy/0]).

-define(MASTER_POOL_NAME(Pool), binary_to_atom(<<Pool/binary, "_master">>, utf8)).
-define(SLAVE_POOL_NAME(Pool), binary_to_atom(<<Pool/binary, "_slave">>, utf8)).
-define(CALL_TIMEOUT, timer:hours(24)).

-define(DEFAULT_GROUP, <<"main_group">>).
-define(DEFAULT_ENVIRONMENT_ID, <<"default_executor">>).


%% Message broadcasted to all workers on pool (see broadcast_job_cancellation/2). 
%% Its existence in message queue determines whether given task was cancelled.
%% Message and matcher structure is determined by worker_pool broadcast message format and its 
%% underlying implementation.
%% As this message is a worker_pool cast message `is_job_cancelled` function will be executed 
%% after job execution is finished, which will result in removal of this message from process message queue.
-define(CANCELLATION_MESSAGE(TaskId), {?MODULE, is_job_cancelled, [TaskId]}).
-define(CANCELLATION_MESSAGE_MATCHER(TaskId), {'$gen_cast', {cast, ?CANCELLATION_MESSAGE(TaskId)}}).

%%%===================================================================
%%% API
%%%===================================================================

%%--------------------------------------------------------------------
%% @doc
%% Inits the pool as an internal service to allow failover.
%% @end
%%--------------------------------------------------------------------
-spec init_pool(pool(), non_neg_integer(), non_neg_integer(), non_neg_integer(), pool_options()) -> ok | no_return().
init_pool(PoolName, MasterJobsNum, SlaveJobsNum, ParallelOrdersLimit, Options) ->
    StartArgs = [PoolName, MasterJobsNum, SlaveJobsNum, ParallelOrdersLimit, Options],
    TakeoverArgs = [PoolName, Options, node()],
    ServiceOptions = #{
        start_function => init_pool_service,
        start_function_args => StartArgs,
        takeover_function => restart_tasks,
        takeover_function_args => TakeoverArgs,
        migrate_function => undefined,
        migrate_function_args => [],
        stop_function => stop_pool_service,
        stop_function_args => [PoolName]
    },
    internal_services_manager:start_service(?MODULE, PoolName, ServiceOptions).

%%--------------------------------------------------------------------
%% @doc
%% Inits the pool (starting appropriate worker pools and adding node to load balancing document)
%% and restarts tasks if needed.
%% @end
%%--------------------------------------------------------------------
-spec init_pool_service(pool(), non_neg_integer(), non_neg_integer(), non_neg_integer(), pool_options()) -> ok | no_return().
init_pool_service(PoolName, MasterJobsNum, SlaveJobsNum, ParallelOrdersLimit, Options) ->
    MasterPool = worker_pool:start_sup_pool(?MASTER_POOL_NAME(PoolName), [{workers, MasterJobsNum}, {queue_type, lifo}]),
    SlavePool = worker_pool:start_sup_pool(?SLAVE_POOL_NAME(PoolName), [{workers, SlaveJobsNum}, {queue_type, lifo}]),
    try
        {{ok, _}, {ok, _}} = {MasterPool, SlavePool}
    catch
        error:{badmatch, {{error, {already_started, _}}, {error, {already_started, _}}}} ->
            throw({error, already_exists})
    end,

    ParallelOrdersPerNodeLimitDefault = max(1, ceil(ParallelOrdersLimit / length(consistent_hashing:get_all_nodes()))),
    ParallelOrdersPerNodeLimit = maps:get(parallel_orders_per_node_limit, Options, ParallelOrdersPerNodeLimitDefault),
    ok = traverse_tasks_scheduler:init(PoolName, ParallelOrdersLimit, ParallelOrdersPerNodeLimit),

    restart_tasks(PoolName, Options, node()).

%%--------------------------------------------------------------------
%% @doc
%% Restart tasks that have been running on particular node. Can be used during node restart or to handle
%% failure of other node.
%% Warning - if node has not been stopped properly, some tasks can be corrupted and impossible to be restarted.
%% @end
%%--------------------------------------------------------------------
-spec restart_tasks(pool(), pool_options(), node()) -> ok | no_return().
restart_tasks(PoolName, Options, Node) ->
    Executor = maps:get(executor, Options, ?DEFAULT_ENVIRONMENT_ID),
    CallbackModules = maps:get(callback_modules, Options),
    LocalNode = node(),
    ok = traverse_tasks_scheduler:reset_node_ongoing_tasks(PoolName, Node),

    TaskIdToCtxMap = repair_ongoing_tasks(PoolName, Executor, Node),
    lists:foreach(fun(CallbackModule) ->
        {JobsPerTask, JobsWitoutCtx, UpdatedTaskIdToCtxMap} =
            get_tasks_jobs(PoolName, CallbackModule, Node, Executor, TaskIdToCtxMap),
        {TasksToRestart, TasksToCancel} = clasiffy_tasks_to_restart_and_cancel(
            UpdatedTaskIdToCtxMap, JobsPerTask, PoolName, CallbackModule, Options, Node),

        traverse_tasks_scheduler:change_node_ongoing_tasks(PoolName, LocalNode, length(TasksToRestart)),
        clean_tasks_and_jobs(UpdatedTaskIdToCtxMap, JobsPerTask, TasksToCancel, JobsWitoutCtx, PoolName, CallbackModule, Node),
        restart_jobs(UpdatedTaskIdToCtxMap, maps:with(TasksToRestart, JobsPerTask), PoolName, CallbackModule, Executor, Node)
    end, CallbackModules),

    schedule_waiting_tasks_if_possible(PoolName, Executor).

%%--------------------------------------------------------------------
%% @doc
%% Stops pool internal service.
%% @end
%%--------------------------------------------------------------------
-spec stop_pool(pool()) -> ok.
stop_pool(PoolName) ->
    internal_services_manager:stop_service(?MODULE, PoolName, <<>>).

%%--------------------------------------------------------------------
%% @doc
%% Stops pool and prevents load balancing algorithm from scheduling tasks on node.
%% Warning: possible races with task scheduling - make sure that there are no tasks waiting to be executed.
%% @end
%%--------------------------------------------------------------------
-spec stop_pool_service(pool()) -> ok.
stop_pool_service(PoolName) ->
    ok = worker_pool:stop_sup_pool(?MASTER_POOL_NAME(PoolName)),
    ok = worker_pool:stop_sup_pool(?SLAVE_POOL_NAME(PoolName)),

    case traverse_tasks_scheduler:clear(PoolName) of
        ok -> ok;
        {error, not_found} -> ok
    end.

%%--------------------------------------------------------------------
%% @doc
%% @equiv run(PoolName, TaskId, Job, #{}).
%% @end
%%--------------------------------------------------------------------
-spec run(pool(), id(), job()) -> ok.
run(PoolName, TaskId, Job) ->
    run(PoolName, TaskId, Job, #{}).

%%--------------------------------------------------------------------
%% @doc
%% Initializes task. The task can be started immediately or scheduled (see traverse_tasks_scheduler.erl).
%% @end
%%--------------------------------------------------------------------
-spec run(pool(), id(), job(), run_options()) -> ok.
run(PoolName, TaskId, Job, Options) ->
    Executor = maps:get(executor, Options, ?DEFAULT_ENVIRONMENT_ID),
    Creator = maps:get(creator, Options, Executor),
    CallbackModule = maps:get(callback_module, Options, binary_to_atom(PoolName, utf8)),
    TaskGroup = maps:get(group_id, Options, ?DEFAULT_GROUP),
    AdditionalData = maps:get(additional_data, Options, #{}),
    MasterJobMode = maps:get(master_job_mode, Options, all),
    ExtendedCtx = get_extended_ctx(CallbackModule, Job),

    {JobStatus, Node, Description} = case Creator =:= Executor of
        true ->
            case traverse_tasks_scheduler:increment_ongoing_tasks_and_choose_node(PoolName) of
                {ok, ChosenNode} ->
                    {on_pool, ChosenNode, #{master_jobs_delegated => 1}};
                {error, limit_exceeded} ->
                    {waiting, undefined, #{}}
            end;
        _ ->
            {waiting, remote, #{}}
    end,

    {ok, JobId} = CallbackModule:update_job_progress(main_job, Job, PoolName, TaskId, JobStatus),
    ok = traverse_task:create(ExtendedCtx, PoolName, CallbackModule, TaskId, Creator, Executor,
        TaskGroup, JobId, Node, Description, AdditionalData, MasterJobMode),

    case Node of
        undefined ->
            ok;
        remote ->
            ok;
        _ ->
            ok = task_callback(CallbackModule, task_started, TaskId, PoolName),
            ok = rpc:call(Node, ?MODULE, run_on_master_pool, [
                PoolName, ?MASTER_POOL_NAME(PoolName), ?SLAVE_POOL_NAME(PoolName),
                CallbackModule, ExtendedCtx, Executor, TaskId, Job, JobId, MasterJobMode])
    end.

%%--------------------------------------------------------------------
%% @doc
%% Executes action connected with remote task change (can start task).
%% @end
%%--------------------------------------------------------------------
-spec on_task_change(task(), environment_id()) -> ok.
on_task_change(Task, Environment) ->
    case traverse_task:on_task_change(Task, Environment) of
        {remote_change, CallbackModule, MainJobId} ->
            case CallbackModule:get_job(MainJobId) of
                {ok, Job, PoolName, _} ->
                    ExtendedCtx = get_extended_ctx(CallbackModule, Job),
                    case traverse_task:on_remote_change(ExtendedCtx, Task, CallbackModule, Environment) of
                        ok ->
                            ok;
                        {ok, remote_cancel, TaskId} ->
                            broadcast_job_cancellation(PoolName, TaskId),
                            task_callback(CallbackModule, on_cancel_init, TaskId, PoolName),
                            ok
                    end;
                {error, not_found} ->
                    ok
            end;
        {run, CallbackModule, MainJobId} ->
            case CallbackModule:get_job(MainJobId) of
                {ok, Job, PoolName, TaskId} ->
                    {ok, #task_execution_info{
                        master_job_mode = MasterJobMode
                    }} = traverse_task:get_execution_info(Task),
                    maybe_run_scheduled_task(
                        PoolName, CallbackModule, TaskId, Task, Environment, Job, MainJobId, MasterJobMode);
                {error, not_found} ->
                    ok
            end;
        ignore ->
            ok
    end.

%%--------------------------------------------------------------------
%% @doc
%% Starts task scheduled on other environment if possible (limit of parallel tasks is not reached).
%% @end
%%--------------------------------------------------------------------
-spec on_job_change(job(), job_id(), pool(), id(), environment_id()) -> ok.
on_job_change(Job, JobId, PoolName, TaskId, Environment) ->
    case traverse_task:get(PoolName, TaskId) of
        {ok, Task} ->
            case traverse_task:is_enqueued(Task) of
                false ->
                    ok;
                true ->
                    {ok, #task_execution_info{
                        callback_module = CallbackModule,
                        executor = Executor,
                        master_job_mode = MasterJobMode
                    }} = traverse_task:get_execution_info(Task),
                    case Executor =:= Environment of
                        true ->
                            maybe_run_scheduled_task(
                                PoolName, CallbackModule, TaskId, Task, Executor, Job, JobId, MasterJobMode);
                        _ ->
                            ok
                    end
            end;
        {error, not_found} ->
            ok
    end.

%%--------------------------------------------------------------------
%% @doc
%% @equiv cancel(PoolName, TaskId, ?DEFAULT_ENVIRONMENT_ID).
%% @end
%%--------------------------------------------------------------------
-spec cancel(pool(), id()) -> ok | {error, term()}.
cancel(PoolName, TaskId) ->
    cancel(PoolName, TaskId, ?DEFAULT_ENVIRONMENT_ID).

%%--------------------------------------------------------------------
%% @doc
%% Cancels task. Prevents jobs waiting in worker pools queues from execution.
%% @end
%%--------------------------------------------------------------------
-spec cancel(pool(), id(), environment_id()) -> ok | {error, term()}.
cancel(PoolName, TaskId, Environment) ->
    case traverse_task:get(PoolName, TaskId) of
        {ok, Task} ->
            {ok, #task_execution_info{
                callback_module = CallbackModule,
                main_job_id = MainJobId
            }} = traverse_task:get_execution_info(Task),
            MainJob = case CallbackModule:get_job(MainJobId) of
                {ok, Job, _, _} -> Job;
                {error, not_found} -> undefined
            end,
            ExtendedCtx = get_extended_ctx(CallbackModule, MainJob),
            {ok, Info} = traverse_task:cancel(ExtendedCtx, PoolName, CallbackModule, TaskId, Environment),
            case Info of
                local_cancel -> 
                    broadcast_job_cancellation(PoolName, TaskId),
                    task_callback(CallbackModule, on_cancel_init, TaskId, PoolName);
                _ -> 
                    ok
            end;
        Other ->
            Other
    end.

%%%===================================================================
%%% Functions executed on pools
%%%===================================================================

%%--------------------------------------------------------------------
%% @doc
%% Executes master job using function provided by callback module. To be executed by worker pool process.
%% Master job is provided with function to enqueue next master jobs during the callback execution if needed.
%% After callback execution, async_master_jobs from master job answer are enqueued. Afterwards, sequential_slave_jobs
%% and next slave_jobs are executed on slave pool and the process awaits for their finish. At the end, master_jobs
%% are enqueued and finish_callback is executed.
%% @end
%%--------------------------------------------------------------------
-spec execute_master_job(pool(), execution_pool(), execution_pool(), callback_module(), ctx(), environment_id(),
    id(), job(), job_id(), master_job_mode()) -> ok.
execute_master_job(PoolName, MasterPool, SlavePool, CallbackModule, ExtendedCtx, Executor,
    TaskId, Job, JobId, MasterJobMode) ->
    try
        MasterJobExtendedArgs = case MasterJobMode of
            single ->
                #{task_id => TaskId}; % callback is not supported when master_job_mode =:= single
            all ->
                MasterJobCallback = prepare_master_callback(PoolName, MasterPool, SlavePool, CallbackModule,
                    ExtendedCtx, Executor, TaskId),
                #{task_id => TaskId, master_job_starter_callback => MasterJobCallback}
        end,

        {ok, MasterAns} = CallbackModule:do_master_job(Job, MasterJobExtendedArgs),
        MasterJobsList = maps:get(master_jobs, MasterAns, []),
        AsyncMasterJobsList = maps:get(async_master_jobs, MasterAns, []),
        SlaveJobsList = maps:get(slave_jobs, MasterAns, []),
        SequentialSlaveJobsList = maps:get(sequential_slave_jobs, MasterAns, []),
        SlaveJobsDelegatedNum = length(SlaveJobsList) + length(lists:flatten(SequentialSlaveJobsList)),

        Description0 = maps:get(description, MasterAns, #{}),
        Description = Description0#{
            slave_jobs_delegated => SlaveJobsDelegatedNum,
            master_jobs_delegated => length(MasterJobsList) + length(AsyncMasterJobsList)
        },

        {ok, _, Canceled} = traverse_task:update_description(ExtendedCtx, PoolName, TaskId, Description),

        {_, NewDescription, Canceled2} = case Canceled of
            true ->
                ok = traverse_task_list:delete_job_link(PoolName, CallbackModule, node(), JobId),
                {ok, _} = CallbackModule:update_job_progress(JobId, Job, PoolName, TaskId, canceled),
                CancelDescription = #{
                    slave_jobs_delegated => -1 * (length(SlaveJobsList) + length(lists:flatten(SequentialSlaveJobsList))),
                    master_jobs_delegated => -1 * (length(MasterJobsList) + length(AsyncMasterJobsList)) - 1
                },
                CancelCallback = maps:get(cancel_callback, MasterAns, fun(_Description) -> ok end),
                CancelCallback(CancelDescription),
                {ok, _, _} = traverse_task:update_description(ExtendedCtx, PoolName, TaskId, CancelDescription);
            _ ->
                ok = run_or_queue_on_master_pool(PoolName, MasterPool, SlavePool, CallbackModule, ExtendedCtx,
                    Executor, TaskId, AsyncMasterJobsList, MasterJobMode, MasterJobMode),

                SequentialSlaveAnswers = sequential_run_on_slave_pool(
                    PoolName, SlavePool, CallbackModule, ExtendedCtx, TaskId, SequentialSlaveJobsList),
                SlaveAnswers = run_on_slave_pool(
                    PoolName, SlavePool, CallbackModule, ExtendedCtx, TaskId, SlaveJobsList),

                SyncJobsToRun = case {MasterJobMode, AsyncMasterJobsList} of
                    {single, [_ | _]} -> none; % Single master job has been already started from AsyncMasterJobsList
                    _ -> MasterJobMode
                end,
                ok = run_or_queue_on_master_pool(PoolName, MasterPool, SlavePool, CallbackModule, ExtendedCtx,
                    Executor, TaskId, MasterJobsList, SyncJobsToRun, MasterJobMode),

                {SlavesOk, SlavesErrors} = lists:foldl(fun
                    ({ok, ok}, {OkSum, ErrorSum}) -> {OkSum + 1, ErrorSum};
                    (_, {OkSum, ErrorSum}) -> {OkSum, ErrorSum + 1}
                end, {0, 0}, SequentialSlaveAnswers ++ SlaveAnswers),

                ok = traverse_task_list:delete_job_link(PoolName, CallbackModule, node(), JobId),
                {ok, _} = CallbackModule:update_job_progress(JobId, Job, PoolName, TaskId, ended),
                Description2 = #{
                    slave_jobs_done => SlavesOk,
                    slave_jobs_failed => SlavesErrors,
                    master_jobs_done => 1
                },
                SlavesDescription = #{
                    slave_jobs_delegated => SlaveJobsDelegatedNum,
                    slave_jobs_done => SlavesOk,
                    slave_jobs_failed => SlavesErrors
                },
                FinishCallback = maps:get(finish_callback, MasterAns, fun(_Args, _SlavesDescription) -> ok end),
                FinishCallback(MasterJobExtendedArgs, SlavesDescription),

                case {AsyncMasterJobsList, MasterJobsList} of
                    {[], []} ->
                        run_from_queue(PoolName, MasterPool, SlavePool, CallbackModule, ExtendedCtx,
                            Executor, TaskId, MasterJobMode);
                    _ ->
                        ok
                end,

                {ok, _, _} = traverse_task:update_description(ExtendedCtx, PoolName, TaskId, Description2)
        end,

        try
            maybe_finish(PoolName, CallbackModule, ExtendedCtx, TaskId, Executor, NewDescription, Canceled2)
        catch
            E2:R2:Stacktrace2 ->
                JobDetails = to_string(CallbackModule, Job),
                ?error_exception(?autoformat([TaskId, CallbackModule, JobDetails]), E2, R2, Stacktrace2)
        end
    catch
        E1:R1:Stacktrace1 ->
            MasterJobDetails = to_string(CallbackModule, Job),
            ?error_exception(?autoformat([TaskId, CallbackModule, MasterJobDetails]), E1, R1, Stacktrace1),
            ErrorDescription = #{
                master_jobs_failed => 1
            },
            % TODO - VFS-5532
            catch traverse_task_list:delete_job_link(PoolName, CallbackModule, node(), JobId),
            catch CallbackModule:update_job_progress(JobId, Job, PoolName, TaskId, failed),
            {ok, ErrorDescription2, Canceled3} = traverse_task:update_description(
                ExtendedCtx, PoolName, TaskId, ErrorDescription),

            try
                maybe_finish(PoolName, CallbackModule, ExtendedCtx, TaskId, Executor, ErrorDescription2, Canceled3)
            catch
                E3:R3:Stacktrace3 ->
                    ?error_exception(?autoformat([TaskId, CallbackModule, MasterJobDetails]), E3, R3, Stacktrace3)
            end
    end,
    ok.

%%--------------------------------------------------------------------
%% @doc
%% Executes slave job using function provided by callback module. To be executed by worker pool process.
%% @end
%%--------------------------------------------------------------------
-spec execute_slave_job(pool(), callback_module(), ctx(), id(), job()) -> ok | error.
execute_slave_job(PoolName, CallbackModule, ExtendedCtx, TaskId, Job) ->
    try
        case traverse_task:is_cancelled(PoolName, TaskId) of
            {ok, true} -> 
                ok;
            {ok, false} ->
                case CallbackModule:do_slave_job(Job, TaskId) of
                    ok ->
                        ok;
                    {ok, Description} ->
                        {ok, _, _} = traverse_task:update_description(ExtendedCtx, PoolName, TaskId, Description),
                        ok;
                    {error, _} ->
                        error
                end;
            {error, _} ->
                error
        end
    catch
        E:R:Stacktrace ->
            SlaveJobDetails = to_string(CallbackModule, Job),
            ?error_exception(?autoformat([TaskId, CallbackModule, SlaveJobDetails]), E, R, Stacktrace),
            error
    end.

%%--------------------------------------------------------------------
%% @doc
%% Based on existence of cancel cast message (see broadcast_job_cancellation/2) determines 
%% whether given task was cancelled. Must be called from slave/master job process. 
%% As this message is a worker_pool cast message this function will be executed after job execution is 
%% finished, which will result in removal of this message from process message queue.
%% @end
%%--------------------------------------------------------------------
-spec is_job_cancelled(id()) -> boolean().
is_job_cancelled(TaskId) ->
    receive
        ?CANCELLATION_MESSAGE_MATCHER(TaskId) = Msg -> 
            % Resend message so next calls to this function from the same job return correct result. 
            % After job execution finish this message will be handled by worker_pool resulting in 
            % its removal from message queue. It is enough as following jobs queued on worker pool 
            % check whether task is cancelled before execution.
            self() ! Msg, 
            true
    after 
        0 -> false
    end.

%%%===================================================================
%%% Internal functions
%%%===================================================================

-spec run_on_slave_pool(pool(), execution_pool(), callback_module(), ctx(), id(), job() | [job()]) -> [ok | error].
run_on_slave_pool(PoolName, SlavePool, CallbackModule, ExtendedCtx, TaskId, Jobs) when is_list(Jobs) ->
    lists_utils:pmap(fun(Job) ->
        worker_pool:call(SlavePool, {?MODULE, execute_slave_job, [PoolName, CallbackModule, ExtendedCtx, TaskId, Job]},
            worker_pool:default_strategy(), ?CALL_TIMEOUT)
    end, Jobs);
run_on_slave_pool(PoolName, SlavePool, CallbackModule, ExtendedCtx, TaskId, Job) ->
    run_on_slave_pool(PoolName, SlavePool, CallbackModule, ExtendedCtx, TaskId, [Job]).

-spec sequential_run_on_slave_pool(pool(), execution_pool(), callback_module(), ctx(), id(), [job() | [job()]]) ->
    [ok | error].
sequential_run_on_slave_pool(PoolName, SlavePool, CallbackModule, ExtendedCtx, TaskId, Jobs) ->
    Ans = lists:map(fun(ParallelJobs) ->
        run_on_slave_pool(PoolName, SlavePool, CallbackModule, ExtendedCtx, TaskId, ParallelJobs)
    end, Jobs),
    lists:flatten(Ans).

-spec run_or_queue_on_master_pool(pool(), execution_pool(), execution_pool(), callback_module(), ctx(), environment_id(),
    id(), [job() | {job(), job_id()}], master_job_mode() | none, master_job_mode()) -> ok.
run_or_queue_on_master_pool(_PoolName, _MasterPool, _SlavePool, _CallbackModule, _ExtendedCtx,
    _Executor, _TaskId, [], _RunNow, _MasterJobMode) ->
    ok;
run_or_queue_on_master_pool(PoolName, MasterPool, SlavePool, CallbackModule, ExtendedCtx,
    Executor, TaskId, Jobs, RunNow, MasterJobMode) ->
    {JobsToRun, JobsToQueue} = case RunNow of
        all -> {Jobs, []};
        single -> {[hd(Jobs)], tl(Jobs)};
        none -> {[], Jobs}
    end,

    lists:foreach(fun
        ({Job, JobId}) ->
            run_on_master_pool(PoolName, MasterPool, SlavePool, CallbackModule,
                ExtendedCtx, Executor, TaskId, Job, JobId, MasterJobMode);
        (Job) ->
            run_on_master_pool(PoolName, MasterPool, SlavePool, CallbackModule,
                ExtendedCtx, Executor, TaskId, Job, undefined, MasterJobMode)
    end, JobsToRun),

    lists:foreach(fun
        ({Job, JobId}) -> queue_on_master_pool(PoolName, CallbackModule, TaskId, Job, JobId);
        (Job) -> queue_on_master_pool(PoolName, CallbackModule, TaskId, Job, undefined)
    end, JobsToQueue).

-spec prepare_master_callback(pool(), execution_pool(), execution_pool(), callback_module(), ctx(), environment_id(),
    id()) -> master_job_starter_callback().
prepare_master_callback(PoolName, MasterPool, SlavePool, CallbackModule, ExtendedCtx, Executor, TaskId) ->
    fun(Args) ->
        Jobs = maps:get(jobs, Args, []),
        Description = #{
            master_jobs_delegated => length(Jobs)
        },
        {ok, _, Canceled} = traverse_task:update_description(ExtendedCtx, PoolName, TaskId, Description),
        case Canceled of
            true ->
                CancelDescription = #{
                    master_jobs_delegated => -1 * length(Jobs)
                },
                case maps:get(cancel_callback, Args, undefined) of
                    undefined -> ok;
                    CancelCallback -> CancelCallback(CancelDescription)
                end,
                {ok, _, _} = traverse_task:update_description(ExtendedCtx, PoolName, TaskId, CancelDescription);
            _ ->
                run_or_queue_on_master_pool(
                    PoolName, MasterPool, SlavePool, CallbackModule, ExtendedCtx, Executor, TaskId, Jobs, all, all)
        end
    end.

-spec run_on_master_pool(pool(), execution_pool(), execution_pool(), callback_module(), ctx(), environment_id(),
    id(), job(), job_id(), master_job_mode()) -> ok.
run_on_master_pool(PoolName, MasterPool, SlavePool, CallbackModule, ExtendedCtx, Executor, TaskId, Job, JobId, MasterJobMode) ->
    {ok, JobId2} = CallbackModule:update_job_progress(JobId, Job, PoolName, TaskId, on_pool),
    ok = traverse_task_list:add_job_link(PoolName, CallbackModule, JobId2),
    ok = worker_pool:cast(MasterPool, {?MODULE, execute_master_job,
        [PoolName, MasterPool, SlavePool, CallbackModule, ExtendedCtx, Executor, TaskId, Job, JobId2, MasterJobMode]}).

-spec queue_on_master_pool(pool(), callback_module(), id(), job(), job_id()) -> ok.
queue_on_master_pool(PoolName, CallbackModule, TaskId, Job, JobId) ->
    {ok, JobId2} = CallbackModule:update_job_progress(JobId, Job, PoolName, TaskId, on_pool),
    ok = traverse_task_list:add_task_job_link(PoolName, CallbackModule, TaskId, JobId2).

-spec run_from_queue(pool(), execution_pool(), execution_pool(), callback_module(), ctx(), environment_id(),
    id(), master_job_mode()) -> ok.
run_from_queue(PoolName, MasterPool, SlavePool, CallbackModule, ExtendedCtx, Executor, TaskId, MasterJobMode) ->
    case traverse_task_list:get_and_delete_first_task_job_link(PoolName, CallbackModule, node(), TaskId) of
        {ok, JobId} ->
            case  CallbackModule:get_job(JobId) of
                {ok, Job, _, _} ->
                    ExtendedCtx = get_extended_ctx(CallbackModule, Job),
                    ok = run_on_master_pool(PoolName, MasterPool, SlavePool, CallbackModule, ExtendedCtx,
                        Executor, TaskId, Job, JobId, MasterJobMode);
                {error, not_found} ->
                    ?warning("Job ~tp not found for pool ~tp, callback module ~tp, task ~tp",
                        [JobId, PoolName, CallbackModule, TaskId])

            end;
        {error, not_found} ->
            ok
    end.

-spec maybe_finish(pool(), callback_module(), ctx(), id(), environment_id(), description(), boolean()) -> ok.
maybe_finish(PoolName, CallbackModule, ExtendedCtx, TaskId, Executor, #{
    master_jobs_delegated := Delegated
} = Description, Canceled) ->
    Done = maps:get(master_jobs_done, Description, 0),
    Failed = maps:get(master_jobs_failed, Description, 0),

    case Delegated == Done + Failed of
        true ->
            % VFS-5532 - can never be equal in case of description saving error
            ok = case Canceled of
                true -> task_callback(CallbackModule, task_canceled, TaskId, PoolName);
                _ -> task_callback(CallbackModule, task_finished, TaskId, PoolName)
            end,

            case traverse_task:finish(ExtendedCtx, PoolName, CallbackModule, TaskId, false, graceful) of
                ok -> check_task_list_and_run(PoolName, Executor, []);
                {error, already_finished} -> ok
            end;
        _ -> ok
    end.

-spec check_task_list_and_run(pool(), environment_id(), [traverse:group()]) -> ok.
check_task_list_and_run(PoolName, Executor, CheckedGroups) ->
    case traverse_tasks_scheduler:get_next_group(PoolName) of
        {error, no_groups} ->
            ok = traverse_tasks_scheduler:decrement_ongoing_tasks(PoolName),
            case traverse_tasks_scheduler:get_next_group(PoolName) of
                {error, no_groups} ->
                    ok;
                _ ->
                    % Race with task starting
                    retry_run(PoolName, Executor, 0)
            end;
        {ok, GroupId} ->
            case lists:member(GroupId, CheckedGroups) of
                true ->
                    ok = traverse_tasks_scheduler:decrement_ongoing_tasks(PoolName),
                    retry_run(PoolName, Executor, 10000);
                false ->
                    StartAns = case traverse_task_list:get_first_scheduled_link(PoolName, GroupId, Executor) of
                        {ok, not_found} ->
                            case deregister_group_and_check(PoolName, GroupId, Executor) of
                                ok ->
                                    start_interrupted;
                                {abort, TaskId} ->
                                    run_task(PoolName, TaskId, Executor)
                            end;
                        {ok, TaskId} ->
                            run_task(PoolName, TaskId, Executor)
                    end,

                    case StartAns of
                        ok ->
                            ok;
                        start_interrupted ->
                            check_task_list_and_run(PoolName, Executor, [GroupId | CheckedGroups])
                    end
            end
    end.

-spec run_task(pool(), id(), environment_id()) -> ok | start_interrupted.
run_task(PoolName, TaskId, Executor) ->
    {ok, #task_execution_info{
        callback_module = CallbackModule,
        main_job_id = MainJobId,
        master_job_mode = MasterJobMode
    }} = traverse_task:get_execution_info(PoolName, TaskId),
    case  CallbackModule:get_job(MainJobId) of
        {ok, Job, _, _} ->
            ExtendedCtx = get_extended_ctx(CallbackModule, Job),
            case traverse_task:start(ExtendedCtx, PoolName, CallbackModule, TaskId, #{master_jobs_delegated => 1}) of
                ok ->
                    ok = task_callback(CallbackModule, task_started, TaskId, PoolName),
                    ok = run_on_master_pool(PoolName, ?MASTER_POOL_NAME(PoolName), ?SLAVE_POOL_NAME(PoolName),
                        CallbackModule, ExtendedCtx, Executor, TaskId, Job, MainJobId, MasterJobMode);
                {error, start_aborted} ->
                    start_interrupted;
                {error, not_found} ->
                    start_interrupted
            end;
        {error, not_found} ->
            start_interrupted
    end.

-spec retry_run(pool(), environment_id(), non_neg_integer()) -> ok.
retry_run(PoolName, Executor, Delay) ->
    spawn(fun() ->
        timer:sleep(Delay),

        case traverse_tasks_scheduler:get_next_group(PoolName) of
            {error, no_groups} ->
                ok;
            {ok, GroupId} ->
                ToStart = case traverse_task_list:get_first_scheduled_link(PoolName, GroupId, Executor) of
                    {ok, not_found} ->
                        case deregister_group_and_check(PoolName, GroupId, Executor) of
                            ok ->
                                empty_group;
                            {abort, ID} ->
                                {ok, ID}
                        end;
                    {ok, ID} ->
                        {ok, ID}
                end,

                case ToStart of
                    empty_group ->
                        retry_run(PoolName, Executor, 0);
                    {ok, TaskId} ->
                        case traverse_tasks_scheduler:increment_ongoing_tasks_and_choose_node(PoolName) of
                            {ok, Node} ->
                                case rpc:call(Node, ?MODULE, run_task, [PoolName, TaskId, Executor]) of
                                    ok ->
                                        ok;
                                    start_interrupted ->
                                        % TODO VFS-6297 - what if node crashes before next line
                                        traverse_tasks_scheduler:decrement_ongoing_tasks(PoolName),
                                        retry_run(PoolName, Executor, 10000)
                                end;
                            {error, limit_exceeded} ->
                                ok
                        end
                end
        end
    end),
    ok.

-spec maybe_run_scheduled_task(pool(), callback_module(), id(), task(), environment_id(),
    job(), job_id(), master_job_mode()) -> ok.
maybe_run_scheduled_task(PoolName, CallbackModule, TaskId, Task, Executor, Job, MainJobId, MasterJobMode) ->
    case traverse_tasks_scheduler:increment_ongoing_tasks_and_choose_node(PoolName) of
        {ok, Node} ->
            ExtendedCtx = get_extended_ctx(CallbackModule, Job),
            case traverse_task:start(ExtendedCtx, PoolName, CallbackModule, TaskId, #{master_jobs_delegated => 1}) of
                ok ->
                    ok = task_callback(CallbackModule, task_started, TaskId, PoolName),
                    ok = rpc:call(Node, ?MODULE, run_on_master_pool, [PoolName, ?MASTER_POOL_NAME(PoolName),
                        ?SLAVE_POOL_NAME(PoolName), CallbackModule, ExtendedCtx, Executor, TaskId, Job,
                        MainJobId, MasterJobMode]);
                {error, start_aborted} ->
                    % TODO VFS-6297 - what if node crashes before next line
                    traverse_tasks_scheduler:decrement_ongoing_tasks(PoolName)
            end;
        {error, limit_exceeded} ->
            traverse_task:schedule_for_local_execution(PoolName, TaskId, Task)
    end.

-spec deregister_group_and_check(pool(), group(), environment_id()) -> ok| {abort, traverse:id()}.
deregister_group_and_check(PoolName, Group, Executor) ->
    ok = traverse_tasks_scheduler:deregister_group(PoolName, Group),
    % check for races with task creation
    case traverse_task_list:get_first_scheduled_link(PoolName, Group, Executor) of
        {ok, not_found} ->
            ok;
        {ok, Task} ->
            traverse_tasks_scheduler:register_group(PoolName, Group),
            {abort, Task}
    end.

-spec get_extended_ctx(callback_module(), job() | undefined) -> ctx().
get_extended_ctx(CallbackModule, undefined) ->
    case erlang:function_exported(CallbackModule, get_sync_info, 1) of
        true ->
            case application:get_env(?CLUSTER_WORKER_APP_NAME, ignore_extended_ctx_get_errors, true) of
                true -> ok;
                false -> ?warning("Getting sync info for undefined job for callback module: ~tp", [CallbackModule])
            end;
        _ ->
            ok
    end,
    traverse_task:get_ctx();
get_extended_ctx(CallbackModule, Job) ->
    {ok, CtxExtension} = case erlang:function_exported(CallbackModule, get_sync_info, 1) of
        true ->
            CallbackModule:get_sync_info(Job);
        _ ->
            {ok, #{}}
    end,
    maps:merge(traverse_task:get_ctx(), CtxExtension).

-spec task_callback(callback_module(), task_callback(), id(), pool()) -> ok.
task_callback(CallbackModule, Method, TaskId, PoolName) ->
    case erlang:function_exported(CallbackModule, Method, 2) of
        true ->
            ok = CallbackModule:Method(TaskId, PoolName);
        _ ->
            ok
    end.

-spec get_node_crash_policy(callback_module(), id(), pool()) -> node_crash_policy().
get_node_crash_policy(CallbackModule, TaskId, PoolName) ->
    case erlang:function_exported(CallbackModule, node_crash_policy, 2) of
        true ->
            CallbackModule:node_crash_policy(TaskId, PoolName);
        _ ->
            restart
    end.

-spec to_string(callback_module(), job()) -> term().
to_string(CallbackModule, Job) ->
    case erlang:function_exported(CallbackModule, to_string, 1) of
        true ->
            CallbackModule:to_string(Job);
        _ -> 
            str_utils:format_bin("~tp", [Job])
    end.

-spec repair_ongoing_tasks(pool(), environment_id(), node()) -> tasks_ctxs().
repair_ongoing_tasks(Pool, Executor, Node) ->
    {ok, TaskIds, _} = traverse_task_list:list(Pool, ongoing, #{tree_id => Executor}),

    % Repair all tasks found using links (links repair is not needed)
    lists:foldl(fun(Id, Acc) ->
        {_, UpdatedAcc} = repair_ongoing_task_and_add_to_map(Pool, Executor, Node, Id, Acc, false),
        UpdatedAcc
    end, #{}, TaskIds).

-spec repair_ongoing_task_and_add_to_map(pool(), environment_id(), node(), id(), tasks_ctxs(), boolean()) ->
    {ok | other_node | not_found, tasks_ctxs()}.
repair_ongoing_task_and_add_to_map(Pool, Executor, Node, Id, TaskIdToCtxMap, FixLink) ->
    case traverse_task:get_execution_info(Pool, Id) of
        {ok, #task_execution_info{
            callback_module = CallbackModule,
            executor = Executor,
            main_job_id = MainJobId,
            node = Node,
            start_time = Timestamp
        }} ->
            case CallbackModule:get_job(MainJobId) of
                {ok, Job, _, _} ->
                    ExtendedCtx = get_extended_ctx(CallbackModule, Job),
                    fix_task_description(Pool, Executor, Node, Id, TaskIdToCtxMap, FixLink, ExtendedCtx, Timestamp);
                {error, not_found} ->
                    ExtendedCtx = get_extended_ctx(CallbackModule, undefined),
                    fix_task_description(Pool, Executor, Node, Id, TaskIdToCtxMap, FixLink, ExtendedCtx, Timestamp);
                JobError ->
                    ?warning("Error getting main job ~tp for task id ~tp (pool ~tp, executor ~tp, node ~tp): ~tp",
                        [MainJobId, Id, Pool, Executor, Node, JobError]),
                    {not_found, TaskIdToCtxMap#{Id => ctx_not_found}}
            end;
        {ok, #task_execution_info{}} ->
            {other_node, TaskIdToCtxMap};
        InfoError ->
            ?warning("Error getting execution info for task id ~tp (pool ~tp, executor ~tp, node ~tp): ~tp",
                [Id, Pool, Executor, Node, InfoError]),
            {not_found, TaskIdToCtxMap}
    end.

-spec fix_task_description(pool(), environment_id(), node(), id(), tasks_ctxs(), boolean(), ctx(), timestamp()) ->
    {ok | other_node, tasks_ctxs()}.
fix_task_description(Pool, Executor, Node, Id, TaskIdToCtxMap, FixLink, ExtendedCtx, Timestamp) ->
    case traverse_task:fix_description(ExtendedCtx, Pool, Id, Node) of
        {ok, _} when FixLink ->
            ok = traverse_task_list:add_link(ExtendedCtx,
                Pool, ongoing, Executor, Id, Timestamp),
            {ok, TaskIdToCtxMap#{Id => ExtendedCtx}};
        {ok, _} ->
            {ok, TaskIdToCtxMap#{Id => ExtendedCtx}};
        {error, other_node} ->
            {other_node, TaskIdToCtxMap}
    end.

-spec get_tasks_jobs(pool(), callback_module(), node(),environment_id(), tasks_ctxs()) ->
    {JobsPerTask :: jobs_per_task(), JobsWitoutCtx :: [job_id()],
        UpdatedTaskIdToCtxMap :: tasks_ctxs()} | no_return().
get_tasks_jobs(PoolName, CallbackModule, Node, Executor, InitialTaskIdToCtxMap) ->
    {ok, JobIds} = traverse_task_list:list_node_jobs(PoolName, CallbackModule, Node),
    lists:foldl(fun(JobId, {JobsPerTask, JobsWitoutCtx, TaskIdToCtxMap}) ->
        case CallbackModule:get_job(JobId) of
            {ok, Job, _, TaskId} ->
                case maps:get(TaskId, TaskIdToCtxMap, ctx_not_found) of
                    ctx_not_found ->
                        ?warning("Found job ~ts assigned to task ~ts (callback module ~tp, pool name ~tp, node ~tp) "
                        "which does not exist (anymore?). Job data:~n~tp~nTrying to find task without link", [
                            JobId, TaskId, CallbackModule, PoolName, Node, Job
                        ]),
                        % Task has not been found using task links so it has not been repaired
                        % Repair it and fix its link
                        case repair_ongoing_task_and_add_to_map(PoolName, Executor, Node, TaskId, TaskIdToCtxMap, true) of
                            {ok, UpdatedTaskIdToCtxMap} ->
                                {JobsPerTask#{TaskId => [{JobId, Job}]}, JobsWitoutCtx, UpdatedTaskIdToCtxMap};
                            {_, UpdatedTaskIdToCtxMap} ->
                                {JobsPerTask, [JobId | JobsWitoutCtx], UpdatedTaskIdToCtxMap}
                        end;
                    _ExtendedCtx ->
                        TaskJobs = maps:get(TaskId, JobsPerTask, []),
                        {JobsPerTask#{TaskId => [{JobId, Job} | TaskJobs]}, JobsWitoutCtx, TaskIdToCtxMap}
                end;
            Error ->
                ?warning("Error getting job ~tp (callback module ~tp, pool name ~tp, node ~tp): ~tp",
                    [JobId, CallbackModule, PoolName, Node, Error]),
                {JobsPerTask, [JobId | JobsWitoutCtx], TaskIdToCtxMap}
        end
    end, {#{}, [], InitialTaskIdToCtxMap}, JobIds).

-spec clasiffy_tasks_to_restart_and_cancel(tasks_ctxs(), jobs_per_task(), pool(), callback_module(),
    pool_options(), node()) -> {TasksToRestart :: [id()], TasksToCancel :: [id()]} | no_return().
clasiffy_tasks_to_restart_and_cancel(TaskIdToCtxMap, JobsPerTask, PoolName, CallbackModule, Options, Node) ->
    ShouldRestart = maps:get(restart, Options, true),
    DBError = datastore_worker:get_application_closing_status() =:= ?CLOSING_PROCEDURE_FAILED,
    LocalNode = node(),
    TasksWithJobs = maps:keys(JobsPerTask),
    OtherTasks = maps:keys(maps:without(TasksWithJobs, TaskIdToCtxMap)),

    case {ShouldRestart, DBError andalso Node =:= LocalNode} of
        {true, true} ->
            lists:foldl(fun(TaskId, {ToRestartAcc, ToCancelAcc}) ->
                case get_node_crash_policy(CallbackModule, TaskId, PoolName) of
                    cancel_task -> {ToRestartAcc, [TaskId | ToCancelAcc]};
                    _ -> {[TaskId | ToRestartAcc], ToCancelAcc}
                end
            end, {[], OtherTasks}, TasksWithJobs);
        {true, false} ->
            {TasksWithJobs, OtherTasks};
        {false, _} ->
            {[], maps:keys(TaskIdToCtxMap)}
    end.

-spec clean_tasks_and_jobs(tasks_ctxs(), jobs_per_task(), [id()], [job_id()],
    pool(), callback_module(), node()) -> ok.
clean_tasks_and_jobs(TaskIdToCtxMap, JobsPerTask, TasksToCancel, JobsWitoutCtx, PoolName, CallbackModule, Node) ->
    lists:foreach(fun(TaskId) ->
        case maps:get(TaskId, TaskIdToCtxMap) of
            ctx_not_found -> traverse_task:finish(traverse_task:get_ctx(), PoolName, CallbackModule, TaskId, true, force);
            ExtendedCtx -> traverse_task:finish(ExtendedCtx, PoolName, CallbackModule, TaskId, true, force)
        end,

        JobsToClean = maps:get(TaskId, JobsPerTask, []),
        clean_jobs(JobsToClean, PoolName, CallbackModule, Node),
        ?info("~tp jobs of task ~tp cleaned - restart impossible", [length(JobsToClean), TaskId])
    end, TasksToCancel),

    clean_jobs(JobsWitoutCtx, PoolName, CallbackModule, Node).

-spec clean_jobs([job_id()], pool(), callback_module(), node()) -> ok.
clean_jobs(JobIds, PoolName, CallbackModule, Node) ->
    lists:foreach(fun(JobId) ->
        traverse_task_list:delete_job_link(PoolName, CallbackModule, Node, JobId)
    end, JobIds).

-spec restart_jobs(tasks_ctxs(), jobs_per_task(), pool(), callback_module(), environment_id(), node()) ->
    ok | no_return().
restart_jobs(TaskIdToCtxMap, JobsPerTask, PoolName, CallbackModule, Executor, Node) ->
    LocalNode = node(),
    lists:foreach(fun({TaskId, JobsToRestart}) ->
        {ok, #task_execution_info{
            master_job_mode = MasterJobMode
        }} = traverse_task:get_execution_info(PoolName, TaskId),
        lists:foreach(fun({JobId, Job}) ->
            case Node of
                LocalNode -> ok;
                _ ->
                    traverse_task_list:add_job_link(PoolName, CallbackModule, JobId),
                    traverse_task_list:delete_job_link(PoolName, CallbackModule, Node, JobId)
            end,

            ExtendedCtx = maps:get(TaskId, TaskIdToCtxMap),
            {ok, _, _} = traverse_task:update_description(ExtendedCtx, PoolName, TaskId, #{
                master_jobs_delegated => 1
            }),
            ok = run_on_master_pool(PoolName, ?MASTER_POOL_NAME(PoolName), ?SLAVE_POOL_NAME(PoolName),
                CallbackModule, ExtendedCtx, Executor, TaskId, Job, JobId, MasterJobMode)
        end, JobsToRestart),

        ?info("~tp jobs of task ~tp restarted", [length(JobsToRestart), TaskId])
    end, maps:to_list(JobsPerTask)).

-spec schedule_waiting_tasks_if_possible(pool(), environment_id()) -> ok | no_return().
schedule_waiting_tasks_if_possible(PoolName, Executor) ->
    case traverse_tasks_scheduler:get_next_group(PoolName) of
        {error, no_groups} ->
            ok;
        {ok, GroupId} ->
            case traverse_task_list:get_first_scheduled_link(PoolName, GroupId, Executor) of
                {ok, not_found} ->
                    deregister_group_and_schedule_waiting_tasks_if_possible(PoolName, Executor, GroupId);
                {ok, TaskId} ->
                    schedule_task_and_check_other_waiting(PoolName, GroupId, Executor, TaskId)
            end
    end.

-spec deregister_group_and_schedule_waiting_tasks_if_possible(pool(), environment_id(), group()) ->
    ok | no_return().
deregister_group_and_schedule_waiting_tasks_if_possible(PoolName, Executor, GroupId) ->
    case deregister_group_and_check(PoolName, GroupId, Executor) of
        ok ->
            ?info("Group ~tp deregistered on restart of node for pool ~tp and executor ~tp",
                [GroupId, PoolName, Executor]),
            schedule_waiting_tasks_if_possible(PoolName, Executor);
        {abort, _TaskId} ->
            ?info("Group ~tp deregistration on restart of node aborted for pool ~tp and executor ~tp",
                [GroupId, PoolName, Executor]),
            schedule_waiting_tasks_if_possible(PoolName, Executor)
    end.

-spec schedule_task_and_check_other_waiting(pool(), group(), environment_id(), id()) -> ok | no_return().
schedule_task_and_check_other_waiting(PoolName, GroupId, Executor, TaskId) ->
    case traverse_tasks_scheduler:increment_ongoing_tasks_and_choose_node(PoolName) of
        {ok, Node} ->
            case rpc:call(Node, ?MODULE, run_task, [PoolName, TaskId, Executor]) of
                ok ->
                    ?info("Task ~tp started on restart of node for pool ~tp and executor ~tp",
                        [TaskId, PoolName, Executor]),
                    schedule_waiting_tasks_if_possible(PoolName, Executor);
                start_interrupted ->
                    ?info("Task ~tp start interrupted on restart of node for pool ~tp and executor ~tp",
                        [TaskId, PoolName, Executor]),
                    traverse_task_list:delete_first_scheduled_link(PoolName, GroupId, Executor, TaskId),
                    % TODO VFS-6297 - what if node crashes before next line
                    traverse_tasks_scheduler:decrement_ongoing_tasks(PoolName),
                    schedule_waiting_tasks_if_possible(PoolName, Executor)
            end;
        {error, limit_exceeded} ->
            ok
    end.

-spec broadcast_job_cancellation(pool(), id()) -> ok.
broadcast_job_cancellation(PoolName, TaskId) ->
    worker_pool:broadcast(?MASTER_POOL_NAME(PoolName), ?CANCELLATION_MESSAGE(TaskId)),
    worker_pool:broadcast(?SLAVE_POOL_NAME(PoolName), ?CANCELLATION_MESSAGE(TaskId)),
    ok.
