%%%-------------------------------------------------------------------
%%% @author Michal Stanisz
%%% @copyright (C) 2020 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% Functions used by ct test to start and manage environment 
%%% using onenv for testing.
%%% @end
%%%-------------------------------------------------------------------
-module(oct_environment).
-author("Michal Stanisz").

-include_lib("ctool/include/test/test_utils.hrl").

%% API
-export([
    setup_environment/2,
    teardown_environment/1,
    disable_panel_healthcheck/1,
    connect_with_nodes/1,
    load_modules/2,
    kill_node/2,
    start_node/2
]).

-define(TIMEOUT, timer:seconds(60)).

-type path() :: string().


%%%===================================================================
%%% API
%%%===================================================================


%% @doc Main function for setting up test environment
-spec setup_environment(test_config:config(), TestModule :: module()) ->
    test_config:config().
setup_environment(Config0, TestModule) ->
    application:start(yamerl),

    BasicConfig = test_config:set_many(read_optional_args(Config0), [
        {set_project_root_path, [test_utils:project_root_dir(Config0)]},
        [test_module, TestModule]
    ]),
    ConfigWithOnenv = init_onenv(BasicConfig),
    ConfigWithCover = oct_coverage:init(ConfigWithOnenv),

    % Start environment and wait for it to be ready
    start_environment(ConfigWithCover),

    % Configure environment
    PodsProplist = proplists:get_value("pods", oct_onenv_cli:status(ConfigWithCover)),
    add_entries_to_etc_hosts(PodsProplist),

    % Setup nodes
    NodesConfig = oct_nodes:refresh_config(ConfigWithCover),
    ConnectedNodesConfig = oct_nodes:connect_with_nodes(NodesConfig),

    % Final configuration
    load_modules(TestModule, ConnectedNodesConfig),

    % when cover is enabled all modules are recompiled resulting in custom configs being lost
    % so setting custom envs manually is necessary
    oct_coverage:is_enabled() andalso set_custom_envs(NodesConfig),

    test_config:set_many(ConnectedNodesConfig, [
        [op_worker_script, script_path(NodesConfig, "op_worker")],
        [cluster_manager_script, script_path(NodesConfig, "cluster_manager")]
    ]).


%% @doc Clean up test environment
-spec teardown_environment(test_config:config()) -> ok.
teardown_environment(Config) ->
    PrivDir = test_config:get_custom(Config, priv_dir),
    CleanEnv = test_config:get_custom(Config, clean_env, true),

    oct_onenv_cli:export_logs(Config, PrivDir),
    CleanEnv andalso test_node_starter:finalize(Config),
    CleanEnv andalso oct_onenv_cli:clean(Config),
    ok.


%% @doc Disable healthcheck for all services
-spec disable_panel_healthcheck(test_config:config()) -> ok.
disable_panel_healthcheck(Config) ->
    oct_nodes:disable_panel_healthcheck(Config).


%% @doc Connect to all nodes in environment
-spec connect_with_nodes(test_config:config()) -> test_config:config().
connect_with_nodes(Config) ->
    oct_nodes:connect_with_nodes(Config).


%% @doc Kill specific node in environment
-spec kill_node(test_config:config(), node()) -> ok.
kill_node(Config, Node) ->
    oct_nodes:kill_node(Config, Node).


%% @doc Start specific node in environment
-spec start_node(test_config:config(), node()) -> ok.
start_node(Config, Node) ->
    oct_nodes:start_node(Config, Node).


%% @doc Load modules on all nodes
-spec load_modules(TestModule :: module(), test_config:config()) -> ok.
load_modules(TestModule, Config) ->
    Modules = [TestModule, test_utils | ?config(load_modules, Config, [])],

    lists:foreach(fun(NodeType) ->
        Nodes = test_config:get_custom(Config, NodeType, []),
        lists:foreach(fun(Node) ->
            lists:foreach(fun(Module) ->
                {Module, Binary, Filename} = code:get_object_code(Module),
                rpc:call(Node, code, delete, [Module], ?TIMEOUT),
                rpc:call(Node, code, purge, [Module], ?TIMEOUT),
                ?assertEqual({module, Module}, rpc:call(
                    Node, code, load_binary, [Module, Filename, Binary], ?TIMEOUT
                ))
            end, Modules)
        end, Nodes)
    end, [cm_nodes, op_worker_nodes, op_panel_nodes, oz_worker_nodes, oz_panel_nodes]).


%%%===================================================================
%%% Internal functions
%%%===================================================================


%% @private
-spec read_optional_args(test_config:config()) -> test_config:config().
read_optional_args(Config) ->
    SourcesFilters = case os:getenv("sources_filters") of
        false -> [];
        SourcesFiltersString -> string:split(SourcesFiltersString, ";")
    end,
    % use utils:ensure_defined as os:getenv accepts only string as default value, so undefined there breaks dialyzer
    OnezoneImage = utils:ensure_defined(os:getenv("onezone_image"), false, undefined),
    OneproviderImage = utils:ensure_defined(os:getenv("oneprovider_image"), false, undefined),
    test_config:set_many(Config, [
        [sources_filters, SourcesFilters],
        [onezone_image, OnezoneImage],
        [oneprovider_image, OneproviderImage]
    ]).


%% @private
-spec init_onenv(test_config:config()) -> test_config:config().
init_onenv(Config) ->
    ProjectRoot = test_config:get_project_root_path(Config),
    OnenvScript = filename:join([ProjectRoot, "one-env", "onenv"]),
    CleanEnv = os:getenv("clean_env"),

    ConfigWithOnenv = test_config:set_many(Config, [
        {set_onenv_script_path, [OnenvScript]},
        [clean_env, CleanEnv == "true"]
    ]),

    % Dummy first call to onenv to setup configs
    PathToSources = os:getenv("path_to_sources"),
    AbsPathToSources = filename:join([ProjectRoot, PathToSources]),
    oct_onenv_cli:status(ConfigWithOnenv, ["--path-to-sources", AbsPathToSources], false, false),

    Sources = oct_onenv_cli:find_sources(ConfigWithOnenv),
    ct:pal("~nUsing sources from:~n~n~ts", [Sources]),

    ConfigWithOnenv.


%% @private
-spec start_environment(test_config:config()) -> ok.
start_environment(Config) ->
    CustomConfigsPaths = add_custom_app_configs(Config),

    ScenarioPath = get_onenv_scenario_path(Config),
    oct_onenv_cli:up(Config, ScenarioPath),
    
    lists:foreach(fun file:delete/1, CustomConfigsPaths),
    
    oct_onenv_cli:await_ready(Config).


%% @private
-spec get_onenv_scenario_path(test_config:config()) -> path().
get_onenv_scenario_path(Config) ->
    ProjectRoot = test_config:get_project_root_path(Config),

    ScenarioName = test_config:get_scenario(Config),
    ?assert(is_list(ScenarioName)),

    ScenarioPath = filename:join([ProjectRoot, "test_distributed", "onenv_scenarios", ScenarioName ++ ".yaml"]),
    ct:pal("Starting onenv scenario ~tp~n~n~tp", [ScenarioName, ScenarioPath]),

    ScenarioPath.


%% @private
-spec add_custom_app_configs(test_config:config()) -> [path()].
add_custom_app_configs(Config) ->
    lists:foldl(fun({Component, Envs}, Acc) ->
        Path = get_test_custom_app_config_path(Config, atom_to_list(Component)),
        file:write_file(Path, io_lib:format("~tp.", [Envs])),
        [Path | Acc]
    end, [], test_config:get_custom_envs(Config)).


%% @private
-spec get_test_custom_app_config_path(test_config:config(), test_config:service_as_list()) -> path().
get_test_custom_app_config_path(Config, Service) ->
    SourcesRelPath = sources_rel_path(Config, Service),
    filename:join([SourcesRelPath, Service, "etc", "config.d", "ct_test_custom.config"]).


%% @private
-spec script_path(test_config:config(), test_config:service_as_list()) -> path().
script_path(Config, Service) ->
    SourcesRelPath = sources_rel_path(Config, Service),
    filename:join([SourcesRelPath, Service, "bin", Service]).


%% @private
-spec sources_rel_path(test_config:config(), test_config:service_as_list()) -> path().
sources_rel_path(Config, Service) ->
    Service1 = re:replace(Service, "_", "-", [{return, list}]),

    OnenvScript = test_config:get_onenv_script_path(Config),
    ProjectRoot = test_config:get_project_root_path(Config),
    SourcesYaml = utils:cmd(["cd", ProjectRoot, "&&", OnenvScript, "find_sources"]),
    [Sources] = yamerl:decode(SourcesYaml),

    filename:join([kv_utils:get([Service1], Sources), "_build", "default", "rel"]).


%% @private
-spec set_custom_envs(test_config:config()) -> ok.
set_custom_envs(Config) ->
    CustomEnvs = test_config:get_custom_envs(Config),
    lists:foreach(fun({Component, Envs}) ->
        ConfigKey = service_to_config_key(Component),
        Nodes = test_config:get_custom(Config, ConfigKey, []),
        lists:foreach(fun(Node) -> set_custom_envs_on_node(Node, Envs) end, Nodes)
    end, CustomEnvs).


%% @private
-spec set_custom_envs_on_node(node(), [{test_config:service(), proplists:proplist()}]) -> ok.
set_custom_envs_on_node(Node, CustomEnvs) ->
    lists:foreach(fun({Application, EnvList}) ->
        lists:foreach(fun({EnvKey, EnvValue}) ->
            ok = rpc:call(Node, application, set_env, [Application, EnvKey, EnvValue])
        end, EnvList)
    end, CustomEnvs).


%% @private
-spec add_entries_to_etc_hosts(proplists:proplist()) -> ok.
add_entries_to_etc_hosts(PodsConfig) ->
    HostsEntries = lists:foldl(fun({_ServiceName, ServiceConfig}, Acc0) ->
        ServiceType = proplists:get_value("service-type", ServiceConfig),
        case lists:member(ServiceType, ["onezone", "oneprovider"]) of
            true ->
                Ip = proplists:get_value("ip", ServiceConfig),

                Acc1 = case proplists:get_value("domain", ServiceConfig, undefined) of
                    undefined -> Acc0;
                    Domain -> [{Domain, Ip} | Acc0]
                end,
                case proplists:get_value("hostname", ServiceConfig, undefined) of
                    undefined -> Acc1;
                    Hostname -> [{Hostname, Ip} | Acc1]
                end;
            false ->
                Acc0
        end
    end, [], PodsConfig),

    {ok, RawFile} = file:read_file("/etc/hosts"),

    EntriesToPreserve = lists:filtermap(fun
        (<<>>) ->
            false;
        (Entry) ->
            [Ip, Domain] = binary:split(Entry, [<<"\t">>, <<" ">>]),
            case Domain of
                <<"dev-one", _Tail/binary>> ->
                    false;
                _ ->
                    {true, {Domain, Ip}}
            end
    end, binary:split(RawFile, <<"\n">>, [global])),
    {ok, File} = file:open("/etc/hosts", [write]),

    lists:foreach(fun({DomainOrHostname, Ip}) ->
        io:fwrite(File, "~ts\t~ts~n", [Ip, DomainOrHostname])
    end, EntriesToPreserve ++ HostsEntries),

    file:close(File).


%% @private
-spec service_to_config_key(test_config:service()) -> test_config:key().
service_to_config_key(op_worker) -> op_worker_nodes;
service_to_config_key(oz_worker) -> oz_worker_nodes;
service_to_config_key(op_panel) -> op_panel_nodes;
service_to_config_key(oz_panel) -> oz_panel_nodes;
service_to_config_key(cluster_manager) -> cm_nodes.
