%%%-------------------------------------------------------------------
%%% @author Bartosz Walkowicz
%%% @copyright (C) 2024 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% Module responsible for interaction with onenv CLI tool.
%%% @end
%%%-------------------------------------------------------------------
-module(oct_onenv_cli).
-author("Bartosz Walkowicz").


%% API
-export([
    % Core commands
    status/1, status/2, status/4,
    up/2,
    await_ready/1,
    clean/1,
    wait/1,
    exec/2,
    
    % Logs & diagnostics
    export_logs/2,
    
    % Sources management
    find_sources/1
]).


%%====================================================================
%% API
%%====================================================================


%% @doc Get deployment status
-spec status(test_config:config()) -> proplists:proplist().
status(Config) ->
    status(Config, []).


-spec status(test_config:config(), [string()]) -> proplists:proplist().
status(Config, OptionalArgs) ->
    status(Config, OptionalArgs, false, true).


-spec status
    (test_config:config(), [string()], boolean(), true) ->proplists:proplist();
    (test_config:config(), [string()], boolean(), false) -> string().
status(Config, OptionalArgs, Print, ParseOutput) ->
    Status = run(Config, ["status"] ++ OptionalArgs),
    Print andalso ct:pal("~ts", [Status]),
    case ParseOutput of
        true ->
            [StatusProplist] = yamerl:decode(Status),
            StatusProplist;
        false ->
            Status
    end.


%% @doc Start environment using onenv up command
-spec up(test_config:config(), file:filename()) -> ok.
up(Config, ScenarioPath) ->
    OptionalArgs = build_onenv_up_optional_args(Config),

    StartCmd = build_onenv_up_command(OptionalArgs, ScenarioPath),
    OnenvStartLogs = run_in_project_root(Config, StartCmd),
    ct:pal("~ts", [OnenvStartLogs]).


%% @private
-spec await_ready(test_config:config()) -> ok.
await_ready(Config) ->
    wait(Config),
    StatusProplist = status(Config, [], true, true),
    case proplists:get_value("ready", StatusProplist) of
        false ->
            export_logs(Config, test_config:get_custom(Config, priv_dir)),
            clean(Config),
            throw(environment_not_ready);
        true ->
            ok
    end.


%% @doc Clean environment and persistent volumes
-spec clean(test_config:config()) -> string().
clean(Config) ->
    run(Config, ["clean", "--all", "--persistent-volumes"]).


%% @doc Wait for deployment to be ready
-spec wait(test_config:config()) -> string().
wait(Config) ->
    run(Config, ["wait", "--timeout", "1200"]).


%% @doc Execute command in pod
-spec exec(test_config:config(), [string()]) -> string().
exec(Config, Args) ->
    run(Config, ["exec"] ++ Args).


%% @doc Export deployment logs
-spec export_logs(test_config:config(), file:filename()) -> ok.
export_logs(Config, PrivDir) ->
    ExportResult = run(Config, ["export", PrivDir]),
    file:write_file(filename:join(PrivDir, "onenv_export.log"), ExportResult).


%% @doc Find sources in project
-spec find_sources(test_config:config()) -> string().
find_sources(Config) ->
    run_in_project_root(Config, ["find_sources"]).


%%====================================================================
%% Internal functions
%%====================================================================


%% @private
-spec run(test_config:config(), [string()]) -> string().
run(Config, Args) ->
    utils:cmd([get_script_path(Config)] ++ Args).


%% @private
-spec run_in_project_root(test_config:config(), [string()]) -> string().
run_in_project_root(Config, Args) ->
    ProjectRoot = test_config:get_project_root_path(Config),
    utils:cmd(["cd", ProjectRoot, "&&", get_script_path(Config)] ++ Args).


%% @private
-spec get_script_path(test_config:config()) -> string().
get_script_path(Config) ->
    test_config:get_onenv_script_path(Config).


%% @private
-spec build_onenv_up_command([string()], string()) -> [string()].
build_onenv_up_command(OptionalArgs, ScenarioPath) ->
    BaseCmd = ["up"] ++ OptionalArgs ++ [ScenarioPath],
    case os:getenv("rsync") of
        "true" -> BaseCmd ++ ["--rsync"];
        _ -> BaseCmd
    end.


%% @private
-spec build_onenv_up_optional_args(test_config:config()) -> [string()].
build_onenv_up_optional_args(Config) ->
    ZoneImageArgs = case test_config:get_custom(Config, onezone_image) of
        undefined  -> [];
        OnezoneImage -> ["-zi", OnezoneImage]
    end,
    ProviderImageArgs = case test_config:get_custom(Config, oneprovider_image) of
        undefined  -> [];
        OneproviderImage -> ["-pi", OneproviderImage]
    end,
    SourcesFilters = test_config:get_custom(Config, sources_filters),
    SourcesFiltersArgs = lists:flatmap(fun(Filter) ->
        ["-sf", Filter]
    end, SourcesFilters),
    ZoneImageArgs ++ ProviderImageArgs ++ SourcesFiltersArgs. 
