%%%--------------------------------------------------------------------
%%% @author Rafal Slota
%%% @copyright (C) 2015 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%--------------------------------------------------------------------
%%% @doc
%%% This module provides error translators for generic fslogic errors.
%%% @end
%%%--------------------------------------------------------------------
-module(fslogic_errors).
-author("Rafal Slota").

-include("global_definitions.hrl").
-include("proto/oneclient/fuse_messages.hrl").
-include("proto/oneclient/common_messages.hrl").
-include("proto/oneclient/proxyio_messages.hrl").
-include("proto/oneprovider/provider_messages.hrl").
-include("modules/fslogic/fslogic_common.hrl").
-include("proto/oneclient/common_messages.hrl").
-include_lib("ctool/include/logging.hrl").

%% API
-export([is_access_error/1]).
-export([handle_error/4, gen_status_message/1]).

% Errors generated by storage_driver.
% Errors that occur in storage_driver should all be reported as one type of error.
-define(SFM_ERROR, filesystem_error).

% This define contains list of error ID and message pairs.
% ErrorMessage should be a human-readable string.
-define(ERROR_DESCRIPTIONS, [
    {?SFM_ERROR, <<"An error occured in underlying file system.">>}
]).

%%%===================================================================
%%% API functions
%%%===================================================================

-spec is_access_error(Reason :: term()) -> boolean().
is_access_error(?EACCES) -> true;
is_access_error(?EPERM) -> true;
is_access_error(?ENOENT) -> true;
is_access_error(not_found) -> true;
is_access_error(_) -> false.

%%--------------------------------------------------------------------
%% @doc
%% Handle error caught during processing of fslogic request.
%% @end
%%--------------------------------------------------------------------
-spec handle_error(fslogic_worker:request(), Type :: atom(), Reason :: term(), stacktrace()) ->
    fslogic_worker:response().
handle_error(Request, Type, Error, Stacktrace) ->
    Status = #status{code = Code} =
        fslogic_errors:gen_status_message(Error),
    LogLevel = code_to_loglevel(Code),
    LogRequest = application:get_env(?CLUSTER_WORKER_APP_NAME,
        log_requests_on_error, false),
    {MsgFormat, FormatArgs} = case LogRequest of
        true ->
            MF = "Cannot process request ~tp (code: ~tp)~nStacktrace: ~ts",
            FA = [lager:pr(Request, ?MODULE), Code,
                iolist_to_binary(lager:pr_stacktrace(Stacktrace, {Type, Error}))],
            {MF, FA};
        _ ->
            MF = "Cannot process request: code: ~tp~nStacktrace: ~ts",
            FA = [Code, iolist_to_binary(lager:pr_stacktrace(Stacktrace, {Type, Error}))],
            {MF, FA}
    end,
    case LogLevel of
        debug -> ?debug(MsgFormat, FormatArgs);
        error -> ?error(MsgFormat, FormatArgs)
    end,
    error_response(Request, Status).

%%--------------------------------------------------------------------
%% @doc
%% Translates operation error to status messages.
%% This function is intended to be extended when new translation is needed.
%% @end
%%--------------------------------------------------------------------
-spec gen_status_message(Error :: term()) -> #status{}.
gen_status_message(?ERROR_UNAUTHORIZED(_)) ->
    #status{code = ?EACCES, description = describe_error(?EACCES)};
gen_status_message(?ERROR_FORBIDDEN) ->
    #status{code = ?EACCES, description = describe_error(?EACCES)};
gen_status_message({error, Reason}) ->
    gen_status_message(Reason);
gen_status_message({badmatch, Error}) ->
    gen_status_message(Error);
gen_status_message({badrpc, Error}) ->
    gen_status_message(Error);
% TODO VFS-5269 - handle other types of processes crashes
gen_status_message({'EXIT', {{Error, _}, _}}) ->
    gen_status_message(Error);
gen_status_message({case_clause, Error}) ->
    gen_status_message(Error);
gen_status_message(#fuse_response{status = Status}) ->
    Status;
gen_status_message({invalid_guid, _}) ->
    #status{code = ?ENOENT, description = describe_error(?ENOENT)};
gen_status_message(not_found) ->
    #status{code = ?ENOENT, description = describe_error(?ENOENT)};
gen_status_message(?MISSING_FILE_META(_MissingUuid)) -> % error thrown by paths_cache
    #status{code = ?ENOENT, description = describe_error(?ENOENT)};
gen_status_message(already_exists) ->
    #status{code = ?EEXIST, description = describe_error(?EEXIST)};
gen_status_message(cancelled) ->
    #status{code = ?ECANCELED, description = describe_error(?ECANCELED)};
gen_status_message(<<"quota exceeded">>) ->
    #status{code = ?ENOSPC, description = describe_error(?ENOSPC)};
gen_status_message({403, <<>>, <<>>}) ->
    #status{code = ?EACCES, description = describe_error(?EACCES)};
gen_status_message(Error) when is_atom(Error) ->
    case errors:is_posix_code(Error) of
        true -> #status{code = Error};
        false -> #status{code = ?EAGAIN, description = describe_error(Error)}
    end;
gen_status_message({ErrorCode, ErrorDescription}) when
    is_atom(ErrorCode) and is_binary(ErrorDescription) ->
    case errors:is_posix_code(ErrorCode) of
        true -> #status{code = ErrorCode, description = ErrorDescription};
        false -> #status{code = ?EAGAIN, description = ErrorDescription}
    end;
gen_status_message(_Reason) ->
    #status{code = ?EAGAIN, description = <<"An unknown error occurred.">>}.

%%%===================================================================
%%% Internal functions
%%%===================================================================

%%--------------------------------------------------------------------
%% @private
%% @doc
%% Convert error code to loglevel.
%% @end
%%--------------------------------------------------------------------
-spec code_to_loglevel(code()) -> error | debug.
code_to_loglevel(?EAGAIN) ->
    error;
code_to_loglevel(_) ->
    debug.

%%--------------------------------------------------------------------
%% @private
%% @doc
%% Returns response with given status, matching given request.
%% @end
%%--------------------------------------------------------------------
-spec error_response(fslogic_worker:request(), #status{}) ->
    #fuse_response{} | #provider_response{} | #proxyio_response{}.
error_response(#fuse_request{}, Status) ->
    #fuse_response{status = Status};
error_response(#file_request{}, Status) ->
    #fuse_response{status = Status};
error_response(#provider_request{}, Status) ->
    #provider_response{status = Status};
error_response(#proxyio_request{}, Status) ->
    #proxyio_response{status = Status}.

%%--------------------------------------------------------------------
%% @private
%% @doc
%% Translates error ID to error description.
%% @end
%%--------------------------------------------------------------------
-spec describe_error(ErrorId :: atom()) -> ErrorDescription :: binary().
describe_error(ErrorId) ->
    case lists:keyfind(ErrorId, 1, ?ERROR_DESCRIPTIONS) of
        {ErrorId, ErrorDescription} -> ErrorDescription;
        false -> atom_to_binary(ErrorId, utf8)
    end.