%%%--------------------------------------------------------------------
%%% @author Krzysztof Trzepla
%%% @copyright (C) 2016 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%--------------------------------------------------------------------
%%% @doc This module is responsible for REST listener starting and stopping.
%%% @end
%%%--------------------------------------------------------------------
-module(rest_listener).
-author("Krzysztof Trzepla").

-include("http/rest.hrl").
-include_lib("ctool/include/logging.hrl").

-export([get_port/0, get_prefix/1]).
-export([start/0, stop/0, status/0]).

%%%===================================================================
%%% API functions
%%%===================================================================

%%--------------------------------------------------------------------
%% @doc Returns REST listener port.
%% @end
%%--------------------------------------------------------------------
-spec get_port() -> Port :: integer().
get_port() ->
    onepanel_env:get(rest_port).


%%--------------------------------------------------------------------
%% @doc Returns REST listener prefix.
%% @end
%%--------------------------------------------------------------------
-spec get_prefix(ApiVersion :: rest_handler:version()) -> Prefix :: binary().
get_prefix(ApiVersion) ->
    Template = onepanel_env:get(rest_api_prefix_template),
    re:replace(Template, "{version_number}",
        onepanel_utils:convert(ApiVersion, binary), [{return, binary}]).


%%--------------------------------------------------------------------
%% @doc Starts REST listener.
%% @end
%%--------------------------------------------------------------------
-spec start() -> ok | no_return().
start() ->
    Port = get_port(),
    HttpsAcceptors = onepanel_env:get(rest_https_acceptors),
    KeyFile = onepanel_env:get(key_file),
    CertFile = onepanel_env:get(cert_file),
    CaCertsDir = onepanel_env:get(cacerts_dir),
    {ok, CaCertPems} = file_utils:read_files({dir, CaCertsDir}),
    CaCerts = lists:map(fun cert_decoder:pem_to_der/1, CaCertPems),

    CommonRoutes = onepanel_api:routes(),
    SpecificRoutes = case onepanel_env:get(release_type) of
        oneprovider -> oneprovider_api:routes();
        onezone -> onezone_api:routes()
    end,
    Routes = merge_routes(CommonRoutes ++ SpecificRoutes) ++ static_gui_routes(),

    Dispatch = cowboy_router:compile([{'_', Routes}]),

    {ok, _} = ranch:start_listener(?MODULE, HttpsAcceptors,
        ranch_ssl, [
            {port, Port},
            {keyfile, KeyFile},
            {certfile, CertFile},
            {cacerts, CaCerts},
            {ciphers, ssl_utils:safe_ciphers()}
        ], cowboy_protocol, [
            {env, [{dispatch, Dispatch}]}
        ]
    ),

    ?info("REST listener successfully started").


%%--------------------------------------------------------------------
%% @doc Stops REST listener.
%% @end
%%--------------------------------------------------------------------
-spec stop() -> ok | {error, Reason :: term()}.
stop() ->
    case ranch:stop_listener(?MODULE) of
        ok ->
            ?info("REST listener stopped");
        {error, Reason} ->
            ?error("Cannot stop REST listener due to: ~p", [Reason]),
            {error, Reason}
    end.


%%--------------------------------------------------------------------
%% @doc Checks whether REST listener is working.
%% @end
%%--------------------------------------------------------------------
-spec status() -> ok | {error, Reason :: term()}.
status() ->
    Endpoint = "https://127.0.0.1:" ++ integer_to_list(get_port()),
    case http_client:get(Endpoint, #{}, <<>>, [insecure]) of
        {ok, _, _, _} -> ok;
        {error, Reason} -> {error, Reason}
    end.

%%%===================================================================
%%% Internal functions
%%%===================================================================

%%--------------------------------------------------------------------
%% @private @doc Converts routes generated by swagger to format expected
%% by cowboy router.
%% @end
%%--------------------------------------------------------------------
-spec merge_routes(Routes) -> Routes when
    Routes :: [{Path :: binary(), Module :: module(), State :: rest_handler:state()}].
merge_routes(Routes) ->
    lists:foldl(fun({Path, Handler, #rstate{methods = [Method]}} = Route, Acc) ->
        case lists:keyfind(Path, 1, Acc) of
            {Path, Handler, #rstate{methods = Methods} = State} ->
                lists:keyreplace(Path, 1, Acc, {Path, Handler,
                    State#rstate{methods = [Method | Methods]}});
            false ->
                [Route | Acc]
        end
    end, [], Routes).


%%--------------------------------------------------------------------
%% @private @doc Returns Cowboy router compliant routes for static gui files.
%% @end
%%--------------------------------------------------------------------
-spec static_gui_routes() -> [{Path :: string(), Module :: module(), State :: term()}].
static_gui_routes() ->
    % Resolve static files root. First, check if there is a non-empty dir
    % located in gui_custom_static_root. If not, use default.
    CustomRoot = onepanel_env:get(gui_custom_static_root),
    DefaultRoot = onepanel_env:get(gui_default_static_root),
    StaticFilesRoot = case file:list_dir_all(CustomRoot) of
        {error, enoent} -> DefaultRoot;
        {ok, []} -> DefaultRoot;
        {ok, _} -> CustomRoot
    end,
   [{"/[...]", gui_static_handler, {dir, StaticFilesRoot}}].
