#include "client.hpp"

#include "common.hpp"
#include "eventBaseHandler.hpp"
#include "httpHeader.hpp"
#include "pingHandler.hpp"
#include "shaper.hpp"
#include "ssl.hpp"

#include <gflags/gflags.h>
#include <wangle/channel/OutputBufferingHandler.h>

DEFINE_uint64(number_of_data_conns, 10, "number of data connections per link");
DEFINE_uint64(
    recv_buf_size, 10 * 1024 * 1024, "size of data conns receive buffer");

using namespace std::literals;

namespace rtransfer {
namespace client {

CtrlIdHandler::CtrlIdHandler(folly::fbstring addr)
    : peerAddr_{std::move(addr)}
{
}

void CtrlIdHandler::read(Context *ctx, folly::IOBufQueue &buf)
{
    if (ctrlIdPromise_.isFulfilled()) {
        ctx->fireRead(buf);
        return;
    }

    if (buf.chainLength() < 18) {
        VLOG(2) << "More data needed to parse ctrlId response from "
                << peerAddr_;
        return;
    }

    VLOG(1) << "Successfuly parsed ctrlId response from " << peerAddr_;

    buf.trimStart(2);  // trim version, we only support 00
    auto ctrlId = buf.split(16)->moveToFbString();
    ctrlIdPromise_.setValue(std::move(ctrlId));
    ctx->fireRead(buf);
}

folly::Future<folly::fbstring> CtrlIdHandler::ctrlId()
{
    return ctrlIdPromise_.getFuture();
}

HttpResponseHandler::HttpResponseHandler(folly::fbstring addr)
    : peerAddr_{std::move(addr)}
{
}

void HttpResponseHandler::read(Context *ctx, folly::IOBufQueue &buf)
{
    if (promise_.isFulfilled()) {
        ctx->fireRead(buf);
        return;
    }

    buf.gather(buf.chainLength());
    if (buf.chainLength() < http::responseLen()) {
        VLOG(2) << "More data needed to parse HTTP response from " << peerAddr_;
        return;
    }

    auto header = buf.split(http::responseLen())->moveToFbString();
    auto res = http::parseResponse(header);
    switch (res.first) {
        case http::ParseStatus::more_data:
            VLOG(2) << "More data needed to parse HTTP response from "
                    << peerAddr_;
            return;
        case http::ParseStatus::bad_header:
            LOG(WARNING) << "Server responded with bad HTTP response "
                         << peerAddr_;
            promise_.setException(
                folly::make_exception_wrapper<std::runtime_error>(
                    "bad header response from server"));
            return;
        case http::ParseStatus::ok:
            VLOG(1) << "Successfuly read HTTP response from " << peerAddr_;
            promise_.setValue();
            ctx->fireRead(buf);
    }
}

folly::Future<folly::Unit> HttpResponseHandler::done()
{
    return promise_.getFuture();
}

void PeerSecretHandler::read(Context *ctx, folly::IOBufQueue &buf)
{
    if (promise_.isFulfilled()) {
        ctx->fireRead(buf);
        return;
    }

    if (buf.chainLength() < proto::secret_size) {
        VLOG(2) << "More data needed to parse peer secret from " << peerAddr_;
        return;
    }

    auto receivedSecret = buf.split(proto::secret_size)->moveToFbString();
    if (peerSecret_ != receivedSecret) {
        LOG(WARNING) << "Bad peer secret received from " << peerAddr_;
        promise_.setException(folly::make_exception_wrapper<std::runtime_error>(
            "bad peer secret"));
        return;
    }

    VLOG(1) << "Successfuly parsed peer secret from " << peerAddr_;
    promise_.setValue();
    ctx->fireRead(buf);
}

folly::Future<folly::Unit> PeerSecretHandler::done()
{
    return promise_.getFuture();
}

PipelineFactory::PipelineFactory(
    std::shared_ptr<ConnectionCloseHandler> connectionCloseHandler,
    folly::fbstring peerSecret, folly::SocketAddress addr)
    : connectionCloseHandler_{std::move(connectionCloseHandler)}
    , peerSecret_{std::move(peerSecret)}
    , peerAddr_{addr.describe()}
{
}

Pipeline::Ptr PipelineFactory::newPipeline(
    std::shared_ptr<folly::AsyncTransportWrapper> sock)
{
    auto pipeline = Pipeline::create();
    pipeline->addBack(wangle::AsyncSocketHandler{sock})
        .addBack(wangle::OutputBufferingHandler{})
        .addBack(EventBaseHandler{})
        .addBack(HttpResponseHandler{peerAddr_})
        .addBack(PeerSecretHandler{peerAddr_, peerSecret_})
        .addBack(CtrlIdHandler{peerAddr_})
        .addBack(connectionCloseHandler_)
        .addBack(wangle::LengthFieldBasedFrameDecoder{2, UINT_MAX, 0, 0, 2})
        .addBack(wangle::LengthFieldPrepender{2})
        .addBack(ProtoHandler<proto::LinkMessage>{})
        .finalize();
    return pipeline;
}

DataPipelineFactory::DataPipelineFactory(
    std::shared_ptr<FetchManager> fetchManager,
    std::shared_ptr<ConnectionCloseHandler> connectionCloseHandler,
    folly::fbstring peerSecret, folly::SocketAddress addr)
    : fetchManager_{std::move(fetchManager)}
    , connectionCloseHandler_{std::move(connectionCloseHandler)}
    , peerSecret_{std::move(peerSecret)}
    , peerAddr_{addr.describe()}
{
}

DataPipeline::Ptr DataPipelineFactory::newPipeline(
    std::shared_ptr<folly::AsyncTransportWrapper> sock)
{
    auto pipeline = DataPipeline::create();
    pipeline->addBack(wangle::AsyncSocketHandler{sock})
        .addBack(EventBaseHandler{})
        .addBack(HttpResponseHandler{peerAddr_})
        .addBack(PeerSecretHandler{peerAddr_, peerSecret_})
        .addBack(PingSender{})
        .addBack(connectionCloseHandler_)
        .addBack(wangle::LengthFieldBasedFrameDecoder{4, UINT_MAX, 17, 0, 0})
        .addBack(fetchManager_)
        .finalize();
    return pipeline;
}

Client::Client(folly::SocketAddress addr,
    std::shared_ptr<folly::IOThreadPoolExecutor> clientExecutor,
    StoragesMap &storages, folly::fbstring mySecret, folly::fbstring peerSecret)
    : addr_{std::move(addr)}
    , storages_{storages}
    , mySecret_{std::move(mySecret)}
    , peerSecret_{std::move(peerSecret)}
    , clientExecutor_{clientExecutor}
{
    ctrl_->pipelineFactory(std::make_shared<PipelineFactory>(
        connectionCloseHandler_, peerSecret_, addr_));
    ctrl_->group(clientExecutor);

    LOG_IF(ERROR, mySecret_.size() != proto::secret_size)
        << "Size of secret " << mySecret_.size()
        << " is different from expected " << proto::secret_size
        << ". The handshake will not finish correctly.";
}

Client::~Client() { closeNow(); }

std::unique_ptr<folly::IOBuf> Client::makeHandshake(folly::StringPiece ctrlId)
{
    auto header = folly::IOBuf::copyBuffer(http::makeHeader("example.com"));
    header->prependChain(folly::IOBuf::wrapBuffer(
        "rtransfer00", proto::header_size + proto::version_size));
    header->prependChain(
        folly::IOBuf::copyBuffer(mySecret_.data(), mySecret_.size()));
    header->prependChain(
        folly::IOBuf::copyBuffer(ctrlId.data(), ctrlId.size()));
    return header;
}

folly::Future<folly::Unit> Client::connect()
{
    if (rtransfer::useSSL())
        ctrl_->sslContext(rtransfer::sslContext());

    return ctrl_->connect(addr_).then([this] {
        auto transport = ctrl_->getPipeline()->getTransport();
        auto base = transport->getEventBase();

        return via(base,
            [this, transport] {
                common::setNoDelay("client ctrl", transport);
                DLOG(INFO) << "Sending control header to " << addr_.describe();
                auto handshake =
                    makeHandshake(folly::fbstring(proto::ctrlid_size, '0'));
                return ctrl_->getPipeline()
                    ->getContext<wangle::LengthFieldPrepender>()
                    ->fireWrite(std::move(handshake));
            })
            .then(base,
                [this] {
                    DLOG(INFO) << "HttpResponseHandler";
                    return ctrl_->getPipeline()
                        ->getHandler<HttpResponseHandler>()
                        ->done();
                })
            .then(base,
                [this] {
                    DLOG(INFO) << "PeerSecretHandler";
                    return ctrl_->getPipeline()
                        ->getHandler<PeerSecretHandler>()
                        ->done();
                })
            .then(base,
                [this] {
                    DLOG(INFO) << "CtrlIdHandler";
                    return ctrl_->getPipeline()
                        ->getHandler<CtrlIdHandler>()
                        ->ctrlId();
                })
            .then(base,
                [this](folly::fbstring ctrlId) {
                    DLOG(INFO) << "Removing finished handlers";
                    ctrl_->getPipeline()
                        ->remove<HttpResponseHandler>()
                        .remove<PeerSecretHandler>()
                        .remove<CtrlIdHandler>()
                        .finalize();
                    return ctrlId;
                })
            .then(&executor_,
                [this](folly::fbstring ctrlId) {
                    DLOG(INFO) << "Starting data conns";
                    ctrlId_ = std::move(ctrlId);
                    dispatcher_.setPipeline(ctrl_->getPipeline());

                    folly::fbvector<folly::Future<folly::Unit>> newDataConns;
                    for (std::size_t i = 0; i < FLAGS_number_of_data_conns; ++i)
                        newDataConns.emplace_back(newDataConn());

                    return folly::collectAll(newDataConns);
                })
            .then(&executor_,
                [this](auto /*futures*/) {
                    pinger_ = std::make_unique<PeriodicHandler>(10s,
                        [s = std::weak_ptr<Client>{this->shared_from_this()}] {
                            if (auto self = s.lock())
                                self->ping();
                        });
                })
            .onError([this](folly::exception_wrapper ew) {
                closeNow();
                return folly::makeFuture<folly::Unit>(std::move(ew));
            });
    });
}

folly::Future<folly::Unit> Client::ping()
{
    VLOG(2) << "Sending ping";
    auto msg = std::make_unique<proto::LinkMessage>();
    msg->mutable_ping();

    return via(&executor_)
        .then([this, msg = std::move(msg)]() mutable {
            return service_(std::move(msg));
        })
        .then([](MsgPtr response) {
            return ensureResponse(
                std::move(response), proto::LinkMessage::kPong);
        })
        .then([](MsgPtr) { VLOG(2) << "Received pong"; });
}

void Client::closeNow()
{
    if (closeNowFlag_.test_and_set())
        return;

    auto pipeline = ctrl_->getPipeline();
    if (!pipeline)
        return;

    auto transport = pipeline->getTransport();
    if (!transport)
        return;

    auto base = transport->getEventBase();
    if (!base)
        return;

    base->runInEventBaseThread(
        [pinger = std::move(pinger_), ctrl = std::move(ctrl_),
            data = std::move(data_)]() mutable {
            pinger = nullptr;
            for (auto &d : data) {
                auto base = d->getPipeline()->getTransport()->getEventBase();
                base->runInEventBaseThread(
                    [d = std::move(d)]() mutable { d = nullptr; });
            }
            ctrl = nullptr;
        });
}

folly::Future<std::size_t> Client::fetch(folly::StringPiece srcStorageId,
    folly::StringPiece srcFileId, folly::StringPiece destStorageId,
    folly::StringPiece destFileId, folly::StringPiece fileGuid,
    std::uint64_t offset, std::size_t size, std::uint8_t priority,
    std::uint64_t reqId, folly::StringPiece transferData,
    folly::Function<void(std::uint64_t, std::size_t)> notifyCb)
{
    auto msg = std::make_unique<proto::LinkMessage>();
    auto req = msg->mutable_fetch();
    req->set_file_id(srcFileId.data(), srcFileId.size());
    req->set_src(srcStorageId.data(), srcStorageId.size());
    req->set_dest(destStorageId.data(), destStorageId.size());
    req->set_offset(offset);
    req->set_size(size);
    req->set_req_id(reqId);
    req->set_priority(priority);
    req->set_transfer_data(transferData.data(), transferData.size());

    auto fetchFuture = fetchManager_->newFetch(reqId, destStorageId.fbstr(),
        destFileId.fbstr(), fileGuid.fbstr(), offset, size,
        std::move(notifyCb));

    return via(&executor_, [this, reqId, msg = std::move(msg),
                               fetchFuture = std::move(fetchFuture)]() mutable {
        auto replyFuture = service_(std::move(msg));
        replyFuture
            .then(&executor_,
                [](MsgPtr response) {
                    return ensureResponse(
                        std::move(response), proto::LinkMessage::kTotalSize);
                })
            .then([this, reqId](MsgPtr msg) {
                fetchManager_->setTotalSize(reqId, msg->total_size());
            })
            .onError([](const folly::FutureCancellation &e) {})
            .onError([this, reqId](folly::exception_wrapper e) {
                fetchManager_->cancelFetch(reqId, std::move(e));
            });

        return fetchFuture
            .ensure([replyFuture = std::move(replyFuture)]() mutable {
                // We have our result, no need to wait for a reply future
                // now If we're on the happy path, the reply's not coming at
                // all
                replyFuture.cancel();
            })
            .via(&executor_);
    });
}

folly::Future<folly::Unit> Client::cancel(std::uint64_t reqId,
    folly::StringPiece srcStorageId, folly::StringPiece destStorageId)
{
    auto msg = std::make_unique<proto::LinkMessage>();
    auto req = msg->mutable_cancel();
    req->set_req_id(reqId);
    req->set_src(srcStorageId.data(), srcStorageId.size());
    req->set_dest(destStorageId.data(), destStorageId.size());

    return via(&executor_)
        .then([this, reqId, msg = std::move(msg)]() mutable {
            fetchManager_->cancelFetch(reqId);
            return service_(std::move(msg));
        })
        .then([](MsgPtr response) {
            return ensureResponse(
                std::move(response), proto::LinkMessage::kDone);
        })
        .then([](MsgPtr) {});
}

void Client::ack(std::uint64_t reqId, std::uint64_t offset)
{
    bool shouldIHandleAcks = false;
    {
        folly::SpinLockGuard guard{acksLock_};
        pendingAcks_.emplace_back(reqId, offset);
        if (!someoneHandlingAcks_)
            shouldIHandleAcks = someoneHandlingAcks_ = true;
    }

    if (shouldIHandleAcks)
        folly::getCPUExecutor()->addWithPriority(
            std::bind(&Client::ackLoop, this), SHAPER_OPS_PRIO);
}

void Client::ackLoop()
{
    while (true) {
        usedAcks_.clear();
        {
            folly::SpinLockGuard guard{acksLock_};
            if (pendingAcks_.empty()) {
                someoneHandlingAcks_ = false;
                return;
            }
            pendingAcks_.swap(usedAcks_);
        }

        std::unordered_map<std::uint64_t, folly::fbvector<std::uint64_t>> byReq;
        for (auto &ack : usedAcks_)
            byReq[ack.first].emplace_back(ack.second);

        auto msg = std::make_unique<proto::LinkMessage>();
        for (auto &ack : byReq) {
            auto a = msg->mutable_acks()->add_acks();
            a->set_req_id(ack.first);
            for (auto offset : ack.second)
                a->add_offsets(offset);
        }

        executor_.addWithPriority(
            [this, msg = std::move(msg)]() mutable {
                service_(std::move(msg)).cancel();
            },
            SHAPER_OPS_PRIO);
    }
}

folly::Future<MsgPtr> Client::ensureResponse(
    MsgPtr msg, proto::LinkMessage::PayloadCase expected)
{
    if (msg->payload_case() == proto::LinkMessage::kError) {
        return folly::makeFuture<MsgPtr>(
            folly::make_exception_wrapper<std::runtime_error>(msg->error()));
    }
    if (msg->payload_case() != expected)
        return folly::makeFuture<MsgPtr>(
            folly::make_exception_wrapper<std::runtime_error>("bad response"));

    return folly::makeFuture<MsgPtr>(std::move(msg));
}

folly::Future<folly::Unit> Client::newDataConn()
{
    VLOG(1) << "Establishing new data connection to " << addr_.describe();
    auto client = std::make_shared<rtransfer::ClientBootstrap<DataPipeline>>();

    if (rtransfer::useSSL())
        client->sslContext(rtransfer::sslContext());

    client->pipelineFactory(std::make_shared<DataPipelineFactory>(
        fetchManager_, connectionCloseHandler_, peerSecret_, addr_));

    folly::AsyncSocket::OptionKey recvBufSize{SOL_SOCKET, SO_RCVBUF};
    client->group(clientExecutor_)
        ->setSocketOptions({{recvBufSize, FLAGS_recv_buf_size}});

    return client->connect(addr_).then([client, this] {
        auto transport = client->getPipeline()->getTransport();
        auto base = transport->getEventBase();

        return via(base,
            [this, transport, client] {
                common::setNoDelay("client data sock", transport);
                VLOG(1) << "Sending data conn header to " << addr_.describe();
                auto handshake = makeHandshake(ctrlId_);
                return client->getPipeline()->write(std::move(handshake));
            })
            .then(base,
                [client] {
                    return client->getPipeline()
                        ->getHandler<HttpResponseHandler>()
                        ->done();
                })
            .then(base,
                [client] {
                    return client->getPipeline()
                        ->getHandler<PeerSecretHandler>()
                        ->done();
                })
            .then(base,
                [client] {
                    client->getPipeline()
                        ->remove<HttpResponseHandler>()
                        .remove<PeerSecretHandler>()
                        .finalize();
                })
            .then(&executor_, [this, client] { data_.emplace_back(client); })
            .onError([this](folly::exception_wrapper ew) {
                closeNow();
                return folly::makeFuture<folly::Unit>(std::move(ew));
            });
    });
}

Client::ClientBootstrapPtr Client::ctrl() { return ctrl_; }
folly::fbvector<DataBootstrapPtr> &Client::data() { return data_; };

}  // namespace client
}  // namespace rtransfer
