#pragma once

#include "proto/rtransfer_messages.pb.h"
#include "protoHandler.hpp"
#include "serverSideLink.hpp"
#include "service.hpp"

#include <folly/io/IOBuf.h>
#include <folly/io/IOBufQueue.h>
#include <wangle/channel/Pipeline.h>
#include <wangle/channel/StaticPipeline.h>
#include <wangle/codec/LengthFieldBasedFrameDecoder.h>
#include <wangle/codec/LengthFieldPrepender.h>
#include <wangle/service/ServerDispatcher.h>
#include <wangle/service/Service.h>

namespace rtransfer {
namespace server {

class ControlHandler : public wangle::HandlerAdapter<folly::IOBufQueue &,
                           std::unique_ptr<folly::IOBuf>> {
public:
    using MsgPtr = std::unique_ptr<proto::LinkMessage>;
    using Pipeline = wangle::Pipeline<folly::IOBufQueue, MsgPtr>;

    ControlHandler(std::shared_ptr<ServerSideLink> link,
        ServiceFactory &serviceFactory, folly::fbstring providerId)
    {
        service_ = serviceFactory(link, std::move(providerId));
        controlPipeline_ = wangle::StaticPipeline<folly::IOBufQueue, MsgPtr,
            ControlHandler, wangle::LengthFieldBasedFrameDecoder,
            wangle::LengthFieldPrepender, ProtoHandler<proto::LinkMessage>,
            ServerSideLink,
            wangle::MultiplexServerDispatcher<MsgPtr, MsgPtr>>::create(this,
            wangle::LengthFieldBasedFrameDecoder{2, UINT_MAX, 0, 0, 2},
            wangle::LengthFieldPrepender{2}, ProtoHandler<proto::LinkMessage>{},
            link,
            wangle::MultiplexServerDispatcher<MsgPtr, MsgPtr>{service_.get()});

        controlPipeline_->setOwner(this);
        ctrlCtx_ = controlPipeline_->getContext<ControlHandler>();
    }

    void attachPipeline(Context *ctx) override { rawCtx_ = ctx; }

    void read(Context *, folly::IOBufQueue &buf) override
    {
        ctrlCtx_->fireRead(buf);
    }

    void readEOF(Context *) override { ctrlCtx_->fireReadEOF(); }

    void readException(Context *, folly::exception_wrapper e) override
    {
        ctrlCtx_->fireReadException(std::move(e));
    }

    void transportActive(Context *) override
    {
        ctrlCtx_->getPipeline()->setTransport(rawCtx_->getTransport());
        ctrlCtx_->fireTransportActive();
    }

    void transportInactive(Context *) override
    {
        ctrlCtx_->fireTransportInactive();
    }

    folly::Future<folly::Unit> write(
        Context *, std::unique_ptr<folly::IOBuf> buf) override
    {
        return rawCtx_->fireWrite(std::move(buf));
    }

    folly::Future<folly::Unit> writeException(
        Context *, folly::exception_wrapper e) override
    {
        return rawCtx_->fireWriteException(std::move(e));
    }

    folly::Future<folly::Unit> close(Context *) override
    {
        return rawCtx_->fireClose();
    }

private:
    Context *rawCtx_{nullptr};
    Context *ctrlCtx_{nullptr};
    Pipeline::Ptr controlPipeline_;
    std::shared_ptr<Service> service_;
};

}  // namespace server
}  // namespace rtransfer
