#pragma once

#include "client.hpp"
#include "common.hpp"
#include "notifyAggregator.hpp"
#include "proto/control.pb.h"
#include "storageHelperFactory.hpp"

#include <folly/Baton.h>
#include <folly/concurrency/ConcurrentHashMap.h>
#include <wangle/channel/Handler.h>
#include <wangle/channel/Pipeline.h>
#include <wangle/service/Service.h>

#include <thread>

namespace rtransfer {
namespace link_control {
namespace proto {
using RequestPtr = std::unique_ptr<Request>;
using ResponsePtr = std::unique_ptr<Response>;
}  // namespace proto

using Pipeline = wangle::Pipeline<folly::IOBufQueue &, proto::ResponsePtr>;

class ControlService
    : public wangle::Service<proto::RequestPtr, proto::ResponsePtr> {
public:
    ControlService(StorageHelperFactory &factory, ClientLinksMap &clientLinks,
        StoragesMap &storages, SecretManager &secretManager)
        : helperFactory_{factory}
        , clientLinks_{clientLinks}
        , storages_{storages}
        , secretManager_{secretManager}
    {
    }

    ~ControlService();

    folly::Future<proto::ResponsePtr> operator()(
        proto::RequestPtr request) override;

    void setPipeline(Pipeline *pipeline) { pipeline_ = pipeline; }

    folly::Future<proto::RequestPtr> ask(proto::ResponsePtr request);

private:
    folly::Future<proto::ResponsePtr> doAction(proto::RequestPtr request);

    static folly::Future<proto::ResponsePtr> error(
        folly::StringPiece description);

    static folly::Future<proto::ResponsePtr> done();

    folly::Future<proto::ResponsePtr> createHelper(proto::RequestPtr req);

    folly::Future<proto::ResponsePtr> connect(proto::RequestPtr req);

    folly::Future<proto::ResponsePtr> fetch(proto::RequestPtr req);

    folly::Future<proto::ResponsePtr> cancel(proto::RequestPtr req);

    folly::Future<proto::ResponsePtr> allowConnection(proto::RequestPtr req);

    folly::Future<proto::ResponsePtr> memoryStats(proto::RequestPtr req);

    void notify(folly::fbstring reqId, std::uint64_t offset, std::size_t size);

    StorageHelperFactory &helperFactory_;
    ClientLinksMap &clientLinks_;
    StoragesMap &storages_;
    folly::ConcurrentHashMap<folly::fbstring,
        std::shared_ptr<folly::Promise<proto::RequestPtr>>>
        questions_;
    std::shared_ptr<folly::IOThreadPoolExecutor> clientExecutor_ =
        std::make_shared<folly::IOThreadPoolExecutor>(
            std::thread::hardware_concurrency(),
            std::make_shared<folly::NamedThreadFactory>("ClientPool"));
    SecretManager &secretManager_;
    Pipeline *pipeline_ = nullptr;

    NotifyAggregator notifyAggregator_{
        [this](folly::fbstring reqId, std::uint64_t offset, std::size_t size) {
            notify(reqId, offset, size);
        }};
};

Pipeline::Ptr newPipeline(ControlService &service, folly::Baton<> &stopBaton);

}  // namespace link_control
}  // namespace rtransfer
