#pragma once

#include <folly/io/IOBuf.h>
#include <google/protobuf/io/zero_copy_stream.h>

#include <memory>

namespace rtransfer {

class IOBufInputStream : public google::protobuf::io::ZeroCopyInputStream {
public:
    IOBufInputStream(std::unique_ptr<folly::IOBuf> buf)
        : _buf{std::move(buf)}
    {
    }

    bool Next(const void **data, int *size) override
    {
        if (_isEnd)
            return false;

        *data = _it->data() + _offset;
        *size = _it->length() - _offset;
        advance();
        return true;
    }

    void BackUp(int count) override
    {
        _it = _it->prev();
        _offset = _it->length() - count;
        _byteCount -= count;
    }

    bool Skip(int count) override
    {
        while (true) {
            if (_isEnd)
                return false;

            if (_it->length() - _offset > count) {
                _offset += count;
                _byteCount += count;
                return true;
            }

            count -= _it->length();
            advance();
        }
    }

    int64_t ByteCount() const override { return _byteCount; }

private:
    void advance()
    {
        _byteCount += _it->length() - _offset;
        _it = _it->next();
        _offset = 0;
        if (_it == _buf.get())
            _isEnd = true;
    }

    bool _isEnd = false;
    std::unique_ptr<folly::IOBuf> _buf;
    folly::IOBuf *_it{_buf.get()};
    std::size_t _offset = 0;
    int64_t _byteCount = 0;
};

} // namespace rtransfer
