#pragma once

#include <folly/Function.h>
#include <folly/executors/GlobalExecutor.h>
#include <folly/io/async/EventBase.h>

#include <chrono>

class PeriodicHandler {
    struct Callback : public folly::HHWheelTimer::Callback {
        template <typename F>
        Callback(std::chrono::milliseconds after, F &&callback)
            : after_{after}
            , onTimeoutExpired_{std::forward<F>(callback)}
            , base_{folly::getIOExecutor()->getEventBase()}
        {
        }

        void timeoutExpired() noexcept override
        {
            onTimeoutExpired_();
            schedule();
        }

        void callbackCanceled() noexcept override {}

        void schedule() noexcept
        {
            base_->runInEventBaseThread(
                [this] { base_->timer().scheduleTimeout(this, after_); });
        }

        std::chrono::milliseconds after_;
        folly::Function<void()> onTimeoutExpired_;
        folly::EventBase *base_;
    };

public:
    template <typename F>
    PeriodicHandler(std::chrono::milliseconds after, F &&callback)
        : callback_{
              std::make_unique<Callback>(after, std::forward<F>(callback))}
    {
        callback_->schedule();
    }

    ~PeriodicHandler()
    {
        callback_->base_->runInEventBaseThreadAndWait(
            [&] { callback_.reset(); });
    }

private:
    std::unique_ptr<Callback> callback_;
};
