#pragma once

#include "ioBufInputStream.hpp"

#include <folly/ExceptionWrapper.h>
#include <folly/io/IOBuf.h>
#include <wangle/channel/Handler.h>
#include <wangle/channel/HandlerContext.h>

#include <exception>
#include <iostream>
#include <memory>

namespace rtransfer {

template <typename Msg>
class ProtoHandler : public wangle::Handler<std::unique_ptr<folly::IOBuf>,
                         std::unique_ptr<Msg>> {
    using typename wangle::Handler<std::unique_ptr<folly::IOBuf>,
        std::unique_ptr<Msg>>::Context;

public:
    virtual void read(Context *ctx, std::unique_ptr<folly::IOBuf> buf) override
    {
        rtransfer::IOBufInputStream stream{std::move(buf)};
        auto msg = std::make_unique<Msg>();
        if (msg->ParseFromZeroCopyStream(&stream)) {
            ctx->fireRead(std::move(msg));
        }
        else {
            ctx->fireReadException(
                folly::make_exception_wrapper<std::runtime_error>(
                    msg->InitializationErrorString()));
        }
    }

    folly::Future<folly::Unit> write(
        Context *ctx, std::unique_ptr<Msg> msg) override
    {
        if (!msg)
            return folly::makeFuture();

        const std::size_t size = msg->ByteSize();
        auto buf = folly::IOBuf::create(size);
        msg->SerializeWithCachedSizesToArray(buf->writableData());
        buf->append(size);
        return ctx->fireWrite(std::move(buf));
    }
};

} // namespace rtransfer
