#include "server.hpp"

#include "common.hpp"
#include "eventBaseHandler.hpp"
#include "httpHeader.hpp"
#include "pingHandler.hpp"
#include "secretManager.hpp"
#include "uuid.hpp"

#include <glog/logging.h>
#include <wangle/channel/OutputBufferingHandler.h>

#include <algorithm>
#include <cctype>

DEFINE_uint64(
    send_buf_size, 10 * 1024 * 1024, "size of data conns send buffer");

DEFINE_string(send_congestion_flavor, "",
    "Set the Flavor of Congestion Control to be used for this Socket. Please "
    "check '/lib/modules/<kernel>/kernel/net/ipv4' for tcp_*.ko first to make "
    "sure the module is available for plugging in. Alternatively you can "
    "choose from net.ipv4.tcp_allowed_congestion_control");

namespace rtransfer {
namespace server {

void RoutingDataHandler::read(Context *ctx, folly::IOBufQueue &bufQueue)

{
    if (isDone_) {
        ctx->fireRead(bufQueue);
        return;
    }

    const_cast<folly::IOBuf *>(bufQueue.front())->coalesce();

    folly::StringPiece data{
        reinterpret_cast<const char *>(bufQueue.front()->data()),
        bufQueue.front()->length()};

    auto res = http::parseHeader(data);
    switch (res.first) {
        case http::ParseStatus::more_data:
            VLOG(2) << "More data needed to parse HTTP header";
            return;
        case http::ParseStatus::bad_header:
            LOG(WARNING) << "Connection rejected due to bad HTTP header";
            closeNow(ctx, "error: bad protocol");
            return;
        case http::ParseStatus::ok:
            VLOG(1) << "HTTP header parsed successfuly";
            if (bufQueue.chainLength() < res.second + proto::handshake_size) {
                VLOG(2) << "More data needed to parse rtransfer handshake";
                return;
            }
            bufQueue.trimStart(res.second);
    }

    auto header = bufQueue.split(proto::header_size)->moveToFbString();
    auto version = bufQueue.split(proto::version_size)->moveToFbString();
    auto secret = bufQueue.split(proto::secret_size)->moveToFbString();

    auto ctrlId = bufQueue.split(proto::ctrlid_size)->moveToFbString();
    bool isControl = false;

    if (ctrlId == folly::fbstring(proto::ctrlid_size, '0')) {
        isControl = true;
        ctrlId = genUUID();
    }

    if (header != "rtransfer") {
        LOG(WARNING) << "Connection rejected due to bad handshake header";
        closeNow(ctx, "error: bad protocol");
        return;
    }

    auto providerIdAndSecretOpt = secretManager_.secretToProviderId(secret);
    if (!providerIdAndSecretOpt) {
        LOG(WARNING) << "Connection rejected due to bad secret";
        closeNow(ctx, "error: unauthorized");
        return;
    }

    folly::fbstring providerId, mySecret;
    std::tie(providerId, mySecret) = *providerIdAndSecretOpt;

    auto pipeline = ctx->getPipeline();

    if (isControl) {
        VLOG(1) << "Accepting new control connection from provider "
                << providerId;
        newControlPipeline(ctx, pipeline, ctrlId, providerId, mySecret);
    }
    else {
        VLOG(1) << "Accepting new data connection from provider " << providerId;
        newDataPipeline(ctx, pipeline, ctrlId, mySecret);
    }

    ctx->getTransport()->getEventBase()->runInLoop(
        [this, pipeline = ctx->getPipelineShared()] {
            pipeline->remove(this).finalize();
        });

    isDone_ = true;
    if (!bufQueue.empty())
        ctx->fireRead(bufQueue);
}

void RoutingDataHandler::closeNow(Context *ctx, folly::StringPiece description)
{
    auto buf = folly::IOBuf::copyBuffer(description, 0, 1);
    *buf->writableTail() = '\n';
    buf->append(1);

    ctx->fireWrite(std::move(buf)).then([ctx] { ctx->fireClose(); });
}

void RoutingDataHandler::newControlPipeline(Context *ctx,
    wangle::PipelineBase *pipeline, folly::StringPiece ctrlId,
    folly::fbstring providerId, folly::StringPiece mySecret)
{
    folly::SocketAddress addr;
    pipeline->getTransport()->getPeerAddress(&addr);
    auto link = linkFactory_(ctrlId.str(), addr.getAddressStr());
    pipeline->addBack(EventBaseHandler())
        .addBack(std::make_shared<ControlHandler>(
            std::move(link), serviceFactory_, std::move(providerId)))
        .finalize();

    common::setNoDelay("server ctrl sock", pipeline->getTransport());

    auto response = folly::IOBuf::copyBuffer(http::makeResponse());
    response->prependChain(folly::IOBuf::copyBuffer(mySecret));
    response->prependChain(
        folly::IOBuf::wrapBuffer("00", 2));  // "negotiated" version
    response->prependChain(folly::IOBuf::copyBuffer(ctrlId));
    ctx->fireWrite(std::move(response));
}

void RoutingDataHandler::newDataPipeline(Context *ctx,
    wangle::PipelineBase *pipeline, const folly::fbstring &ctrlId,
    folly::StringPiece mySecret)
{
    auto dataConnHandler = linkFactory_.createDataConn(ctrlId);
    if (!dataConnHandler) {
        LOG(INFO) << "Rejecting data connection: control connection not found";
        closeNow(ctx, "error: control connection not found");
        return;
    }

    auto sock =
        common::setNoDelay("server data sock", pipeline->getTransport());

    if (!FLAGS_send_congestion_flavor.empty()) {
        int error = sock->setCongestionFlavor(FLAGS_send_congestion_flavor);
        LOG_IF(WARNING, error)
            << "Failed to set congestion flavor on server sock to "
            << FLAGS_send_congestion_flavor << ": " << google::StrError(error);
    }

    pipeline->addBack(PingReceiver{})
        .addBack(wangle::OutputBufferingHandler{})
        .addBack(dataConnHandler)
        .addBack(EventBaseHandler{})
        .finalize();

    auto response = folly::IOBuf::copyBuffer(http::makeResponse());
    response->prependChain(folly::IOBuf::copyBuffer(mySecret));
    ctx->fireWrite(std::move(response));
}

wangle::DefaultPipeline::Ptr PipelineFactory::newPipeline(
    std::shared_ptr<folly::AsyncTransportWrapper> sock)
{
    auto pipeline = wangle::DefaultPipeline::create();
    pipeline->addBack(wangle::AsyncSocketHandler(sock))
        .addBack(
            RoutingDataHandler{linkFactory_, serviceFactory_, secretManager_})
        .finalize();
    return pipeline;
}

std::unique_ptr<wangle::ServerBootstrap<>> makeServer(std::uint16_t port,
    ServerSideLinkFactory linkFactory, ServiceFactory &serviceFactory,
    SecretManager &secretManager,
    wangle::ServerSocketConfig &serverSocketConfig)
{
    folly::AsyncSocket::OptionKey sendBufSize{SOL_SOCKET, SO_SNDBUF};
    // setSocketOptions needs to have a bindAddress set to an ipv4 or ipv6 addr
    // to filter out some options that are not relevant here
    serverSocketConfig.bindAddress = folly::SocketAddress{"0.0.0.0", port};
    serverSocketConfig.setSocketOptions({{sendBufSize, FLAGS_send_buf_size}});
    serverSocketConfig.bindAddress = folly::SocketAddress{};

    auto pipelineFactory = std::make_shared<PipelineFactory>(
        std::move(linkFactory), serviceFactory, secretManager);
    auto server = std::make_unique<wangle::ServerBootstrap<>>();
    auto executor = std::make_shared<folly::IOThreadPoolExecutor>(
        std::thread::hardware_concurrency());
    server->childPipeline(pipelineFactory)
        ->acceptorConfig(serverSocketConfig)
        ->group(executor)
        ->bind(port);
    return server;
}

}  // namespace server
}  // namespace rtransfer
