#pragma once

#include "controlHandler.hpp"
#include "service.hpp"

#include <wangle/bootstrap/ServerBootstrap.h>
#include <wangle/channel/Handler.h>
#include <wangle/channel/Pipeline.h>

namespace rtransfer {
namespace server {

class RoutingDataHandler : public wangle::HandlerAdapter<folly::IOBufQueue &,
                               std::unique_ptr<folly::IOBuf>> {
public:
    RoutingDataHandler(ServerSideLinkFactory &factory,
        ServiceFactory &serviceFactory, SecretManager &secretManager)
        : linkFactory_{factory}
        , serviceFactory_{serviceFactory}
        , secretManager_{secretManager}
    {
    }

    void read(Context *ctx, folly::IOBufQueue &bufQueue) override;

    folly::Future<folly::Unit> write(
        Context *ctx, std::unique_ptr<folly::IOBuf> buf) override
    {
        return ctx->fireWrite(std::move(buf));
    }

private:
    void closeNow(Context *ctx, folly::StringPiece description);

    void newControlPipeline(Context *ctx, wangle::PipelineBase *pipeline,
        folly::StringPiece ctrlId, folly::fbstring providerId,
        folly::StringPiece mySecret);

    void newDataPipeline(Context *ctx, wangle::PipelineBase *pipeline,
        const folly::fbstring &ctrlId, folly::StringPiece mySecret);

    bool isDone_ = false;
    ServerSideLinkFactory &linkFactory_;
    ServiceFactory &serviceFactory_;
    SecretManager &secretManager_;
};

class PipelineFactory
    : public wangle::PipelineFactory<wangle::DefaultPipeline> {
public:
    using MsgPtr = std::unique_ptr<proto::LinkMessage>;

    PipelineFactory(ServerSideLinkFactory factory,
        ServiceFactory &serviceFactory, SecretManager &secretManager)
        : linkFactory_{std::move(factory)}
        , serviceFactory_{serviceFactory}
        , secretManager_{secretManager}
    {
    }

    wangle::DefaultPipeline::Ptr newPipeline(
        std::shared_ptr<folly::AsyncTransportWrapper> sock) override;

private:
    ServerSideLinkFactory linkFactory_;
    ServiceFactory &serviceFactory_;
    SecretManager &secretManager_;
};

std::unique_ptr<wangle::ServerBootstrap<>> makeServer(std::uint16_t port,
    ServerSideLinkFactory linkFactory, ServiceFactory &serviceFactory,
    SecretManager &secretManager,
    wangle::ServerSocketConfig &serverSocketConfig);

}  // namespace server
}  // namespace rtransfer
