#include "serverSideLink.hpp"

#include <gflags/gflags.h>

DEFINE_uint64(throughput_probe_interval, 3000,
    "minimum time in milliseconds after which connections will be reprobed to "
    "check throughput for the purpose of weighted bandwidth splitting");

using namespace std::literals::chrono_literals;

namespace rtransfer {

ServerSideLink::ServerSideLink(std::string peerAddr)
    : peerAddr_{std::move(peerAddr)}
{
}

std::string ServerSideLink::getPeerAddr() { return peerAddr_; }

void ServerSideLink::attachPipeline(Context *ctx) { ctrlCtx_ = ctx; }

void ServerSideLink::addDataConnection(wangle::DefaultPipeline::Ptr pipeline)
{
    auto tInfo = probe(pipeline);
    folly::SpinLockGuard guard{lock_};
    dataConns_.emplace_back(std::move(pipeline), tInfo);
}

void ServerSideLink::removeDataConnection(wangle::DefaultPipeline::Ptr pipeline)
{
    LOG(INFO) << "DATA conn down";
    folly::SpinLockGuard guard{lock_};
    auto it = std::find_if(dataConns_.begin(), dataConns_.end(),
        [&](const auto &el) { return el.first == pipeline; });
    if (it != dataConns_.end()) {
        std::swap(*it, *dataConns_.rbegin());
        dataConns_.pop_back();
    }
}

folly::fbvector<ServerSideLink::Conn> ServerSideLink::dataConnections(
    std::chrono::steady_clock::time_point now)
{
    folly::SpinLockGuard guard{lock_};
    maybeProbe(now);
    return dataConns_;
}

void ServerSideLink::readEOF(Context *ctx)
{
    folly::SpinLockGuard guard{lock_};
    for (auto &conn : dataConns_)
        conn.first->getTransport()->getEventBase()->runInEventBaseThread(
            [conn = conn.first] { conn->getTransport()->close(); });
    ctx->fireReadEOF();
}

void ServerSideLink::maybeProbe(std::chrono::steady_clock::time_point now)
{
    if (now - lastProbe_ <
        std::chrono::milliseconds(FLAGS_throughput_probe_interval))
        return;

    lastProbe_ = now;
    for (auto &conn : dataConns_)
        conn.second = probe(conn.first);
}

wangle::TransportInfo ServerSideLink::probe(wangle::DefaultPipeline::Ptr conn)
{
    wangle::TransportInfo tInfo;
    auto sock = dynamic_cast<folly::AsyncSocket *>(conn->getTransport().get());
    DCHECK(sock);
    tInfo.initWithSocket(sock);
    return tInfo;
}

DataConn::DataConn(std::shared_ptr<ServerSideLink> link)
    : link_{link}
{
}

void DataConn::attachPipeline(Context *ctx)
{
    link_->addDataConnection(getPipeline(ctx));
}

void DataConn::readEOF(Context *ctx)
{
    link_->removeDataConnection(getPipeline(ctx));
    ctx->fireReadEOF();
}

wangle::DefaultPipeline::Ptr DataConn::getPipeline(Context *ctx)
{
    return std::static_pointer_cast<wangle::DefaultPipeline>(
        ctx->getPipelineShared());
}

ServerSideLinkFactory::ServerSideLinkFactory(
    folly::ConcurrentHashMap<folly::fbstring, std::weak_ptr<ServerSideLink>>
        &map)
    : map_{map}
{
}

std::shared_ptr<ServerSideLink> ServerSideLinkFactory::operator()(
    folly::StringPiece ctrlId, std::string peerAddr)
{
    auto link = std::make_shared<ServerSideLink>(std::move(peerAddr));
    map_.insert_or_assign(ctrlId.fbstr(), link);
    return link;
}

std::shared_ptr<DataConn> ServerSideLinkFactory::createDataConn(
    const folly::fbstring &ctrlId)
{
    auto it = map_.find(ctrlId);
    if (it == map_.cend())
        return {};

    if (auto link = it->second.lock())
        return std::make_shared<DataConn>(link);

    map_.erase(ctrlId);
    return {};
}

}  // namespace rtransfer
