#include "service.hpp"

#include "serverSideLink.hpp"
#include "storage.hpp"

#include <monitoring/monitoring.h>

#include <folly/Bits.h>

namespace rtransfer {

folly::Future<Service::MsgPtr> Service::operator()(MsgPtr request)
{
    VLOG(1) << "Request from remote: " << request->DebugString();

    ONE_METRIC_COUNTER_INC("requests.remote.incoming.total");
    ONE_METRIC_COUNTER_INC("requests.remote.incoming.concurrent");

    auto prio = request->payload_case() == proto::LinkMessage::kAcks
                    ? SHAPER_OPS_PRIO
                    : 0;

    auto reqId = request->req_id();
    return via(folly::getCPUExecutor().get(), prio)
        .then([this, request = std::move(request)]() mutable {
            return action(std::move(request));
        })
        .onError([](const std::exception &e) { return error(e.what()); })
        .then([reqId = std::move(reqId)](MsgPtr resp) {
            if (resp) {
                resp->set_req_id(reqId);
                VLOG(1) << "Replying: " << resp->DebugString();
            }

            return resp;
        })
        .ensure([] {
            ONE_METRIC_COUNTER_DEC("requests.remote.incoming.concurrent");
        });
}

folly::Future<Service::MsgPtr> Service::action(MsgPtr request)
{
    switch (request->payload_case()) {
        case proto::LinkMessage::kFetch:
            ONE_METRIC_COUNTER_INC("requests.remote.incoming.fetch");
            return fetch(std::move(request));
        case proto::LinkMessage::kCancel:
            ONE_METRIC_COUNTER_INC("requests.remote.incoming.cancel");
            return cancel(std::move(request));
        case proto::LinkMessage::kAcks:
            return ack(std::move(request));
        case proto::LinkMessage::kPing:
            return ping(std::move(request));
        default:
            ONE_METRIC_COUNTER_INC("requests.remote.incoming.bad_request");
            return error("invalid operation");
    }
}

folly::Future<Service::MsgPtr> Service::fetch(MsgPtr request)
{
    auto reqId = request->fetch().req_id();
    auto requestedSize = request->fetch().size();
    folly::fbstring transferData = request->fetch().transfer_data();
    return authCache_.authorize(providerId_, std::move(transferData))
        .then([reqId, request = std::move(request)](
                  std::shared_ptr<link_control::proto::Request>
                      authResponse) mutable {
            auto &ar = authResponse->auth_response();
            VLOG(1) << "ReqId: " << reqId << " using file_id: " << ar.file_id()
                    << ", storage_id: " << ar.storage_id()
                    << " as source for guid: " << ar.file_guid();

            auto f = request->mutable_fetch();
            f->set_file_id(ar.file_id());
            f->set_src(ar.storage_id());
            f->set_file_guid(ar.file_guid());

            return folly::makeFuture(std::move(request));
        })
        .then([this](MsgPtr request) {
            return shaperMap_.fetch(std::move(request));
        })
        .then([reqId, requestedSize](std::size_t size) {
            if (size >= requestedSize)
                return MsgPtr{nullptr};  // no need to correct client

            VLOG(2) << "Sending back read result req_id: " << reqId
                    << ", size: " << size << " bytes";
            auto resp = std::make_unique<proto::LinkMessage>();
            resp->set_total_size(size);
            return resp;
        })
        .onError(
            [reqId](const folly::AsyncSocketException &e) {
                LOG(WARNING)
                    << "Refusing to confirm req_iq: " << reqId << " due to "
                    << e.what()
                    << ". The connection should soon be dropped by the client "
                       "side and all its requests will be restarted.";
                return std::unique_ptr<proto::LinkMessage>{};
            });
}

folly::Future<Service::MsgPtr> Service::cancel(MsgPtr request)
{
    return shaperMap_.cancel(std::move(request)).then([] {
        auto resp = std::make_unique<proto::LinkMessage>();
        resp->set_done(true);
        return folly::makeFuture<MsgPtr>(std::move(resp));
    });
}

folly::Future<Service::MsgPtr> Service::ack(MsgPtr request)
{
    shaperMap_.ack(std::move(request));
    return folly::makeFuture<MsgPtr>(MsgPtr{nullptr});
}

folly::Future<Service::MsgPtr> Service::ping(MsgPtr)
{
    VLOG(2) << "Received ping; sending pong";
    auto resp = std::make_unique<proto::LinkMessage>();
    resp->mutable_pong();
    return folly::makeFuture<MsgPtr>(std::move(resp));
}

folly::Future<Service::MsgPtr> Service::error(folly::StringPiece description)
{
    auto msg = std::make_unique<proto::LinkMessage>();
    msg->set_error(description.data(), description.size());
    return folly::makeFuture(std::move(msg));
}

}  // namespace rtransfer
