#include "shaperMap.hpp"

#include <gflags/gflags.h>

#include <functional>

namespace rtransfer {

ShaperMap::ShaperMap(Reader &reader)
    : reader_{reader}
{
}

folly::Future<std::size_t> ShaperMap::fetch(MsgPtr msg)
{
    const auto &fetch = msg->fetch();
    const auto reqId = fetch.req_id();

    std::tuple<folly::fbstring, folly::fbstring, std::uint8_t> key{
        fetch.src(), fetch.dest(), fetch.priority()};

    auto res = shapers_.emplace(
        std::move(key), std::make_shared<Shaper>(reader_, shaperTimer_));

    reqToShaper_.emplace(reqId, res.first->second);
    return res.first->second->read(std::move(msg))
        .via(folly::getCPUExecutor().get());
}

folly::Future<folly::Unit> ShaperMap::cancel(MsgPtr msg)
{
    const auto &cancel = msg->cancel();
    auto it = reqToShaper_.find(cancel.req_id());
    if (it == reqToShaper_.cend())
        return folly::makeFuture();

    reqToShaper_.erase(it);
    return it->second->cancel(std::move(msg))
        .via(folly::getCPUExecutor().get());
    ;
}

void ShaperMap::ack(MsgPtr msg)
{
    for (auto &ack : msg->acks().acks()) {
        folly::fbvector<std::uint64_t> offsets;
        offsets.insert(
            offsets.end(), ack.offsets().begin(), ack.offsets().end());

        auto it = reqToShaper_.find(ack.req_id());
        if (it == reqToShaper_.cend()) {
            LOG(WARNING) << "Cannot find shaper for an ack reqId: "
                         << ack.req_id();
            return;
        }
        it->second->ack(ack.req_id(), std::move(offsets))
            .via(folly::getCPUExecutor().get())
            .then([this, reqId = ack.req_id()](bool ackedAll) {
                if (ackedAll)
                    reqToShaper_.erase(reqId);
            });
    }
}

}  // namespace rtransfer
