#pragma once

#include "chainQueue.hpp"
#include "openType.hpp"

#include <folly/FBString.h>
#include <folly/Function.h>
#include <folly/Synchronized.h>
#include <folly/executors/CPUThreadPoolExecutor.h>
#include <folly/executors/NamedThreadFactory.h>
#include <folly/executors/SerialExecutor.h>
#include <folly/io/IOBufQueue.h>
#include <helpers/storageHelper.h>

#include <atomic>
#include <chrono>
#include <memory>

namespace rtransfer {

class Storage {
public:
    struct ReadRequest {
        std::uint64_t reqId;
        folly::Promise<folly::IOBufQueue> promise;
        one::helpers::FileHandlePtr handle;
        std::uint64_t offset;
        std::size_t size;

        std::size_t getSize() const { return size; }
    };

    struct WriteRequest {
        std::uint64_t reqId;
        folly::Promise<folly::Unit> promise;
        one::helpers::FileHandlePtr handle;
        std::uint64_t offset;
        folly::IOBufQueue buf;

        std::size_t getSize() const { return buf.chainLength(); }
    };

    struct Bucket {
        template <typename Request>
        struct Queue {
            std::mutex mutex;
            bool isUsed = false;
            ChainQueue<Request> impl;
        };

        Bucket(std::shared_ptr<folly::Executor> executor_)
            : executor{executor_}
        {
        }

        folly::SerialExecutor executor;
        Queue<ReadRequest> readQueue;
        Queue<WriteRequest> writeQueue;
    };

    Storage(folly::fbstring storageId,
        one::helpers::StorageHelperPtr storageHelper);

    folly::Future<folly::IOBufQueue> read(one::helpers::FileHandlePtr,
        std::uint64_t reqId, const std::uint64_t offset, const std::size_t size,
        const std::uint8_t priority);

    folly::Future<folly::Unit> write(one::helpers::FileHandlePtr,
        std::uint64_t reqId, const std::uint64_t offset, folly::IOBufQueue buf,
        std::uint8_t priority);

    folly::Future<one::helpers::FileHandlePtr> open(
        folly::fbstring fileId, OpenType openType);

private:
    std::string m(const char *);

    template <typename Req, typename Cb>
    auto inBucket(Req &&req, Cb &&cb);

    folly::fbstring storageIdHex_;
    one::helpers::StorageHelperPtr storageHelper_;

    constexpr static const std::size_t maxInProgress_{100};

    folly::fbvector<std::unique_ptr<Bucket>> buckets_;
};

}  // namespace rtransfer
