%%%-------------------------------------------------------------------
%%% @author Konrad Zemek
%%% @copyright (C) 2018 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% @end
%%%-------------------------------------------------------------------
-module(rtransfer_link_cluster).
-author("Konrad Zemek").

-behaviour(gen_server).

%%%===================================================================
%%% Type definitions
%%%===================================================================

-type state() :: no_state.

%%%===================================================================
%%% Exports
%%%===================================================================

-export([start_link/0, fetch/4, cancel/1, set_provider_nodes/2, add_storage/3,
         allow_connection/4]).
-export([init/1, handle_info/2, handle_cast/2, code_change/3, terminate/2, handle_call/3]).

%%%===================================================================
%%% API
%%%===================================================================

-spec start_link() -> {ok, pid()} | {error, any()}.
start_link() ->
    gen_server2:start_link({local, ?MODULE}, ?MODULE, {}, []).

-spec fetch(rtransfer_request:t(), TransferData :: binary(),
            Notify :: rtransfer_link:notify_fun(),
            OnComplete :: rtransfer_link:on_complete_fun()) ->
                   {ok, reference()} | {error, Reason :: any()}.
fetch(Request, TransferData, Notify, OnComplete0) ->
    Node = random_node(),
    OnComplete = fun(Ref, State) ->
                         rtransfer_link_table:delete(Ref),
                         OnComplete0(Ref, State)
                 end,
    case rtransfer_link:fetch({rtransfer_link, Node}, Request,
                              TransferData, Notify, OnComplete) of
        {ok, Ref} ->
            rtransfer_link_table:write(#{ref => Ref, node => Node,
                                         request => Request}),
            {ok, Ref};
        Other ->
            Other
    end.

-spec cancel(reference()) -> ok.
cancel(Ref) ->
    #{node := Node} = rtransfer_link_table:read(Ref),
    gen_server2:call({rtransfer_link, Node}, {cancel, Ref}, infinity).

-spec set_provider_nodes(Nodes :: [node()], CallbackModule :: module()) -> any().
set_provider_nodes(Nodes, CallbackModule) ->
    rpc:multicall(rtransfer_link_table:nodes(), rtransfer_link, set_provider_nodes,
                  [Nodes, CallbackModule]).

-spec add_storage(StorageId :: binary(), HelperName :: binary(),
                  HelperArgs :: [{binary(), binary()}]) -> ok.
add_storage(StorageId, HelperName, HelperArgs) ->
    gen_server2:abcast(rtransfer_link_table:nodes(), rtransfer_link,
                       {add_storage, StorageId, HelperName, HelperArgs}),
    ok.

-spec allow_connection(ProviderId :: binary(), MySecret :: binary(),
                       PeerSecret :: binary(), Expiration :: non_neg_integer()) -> ok.
allow_connection(ProviderId, MySecret, PeerSecret, Expiration) ->
    gen_server2:multi_call(rtransfer_link_table:nodes(), rtransfer_link,
                           {allow_connection, ProviderId, MySecret, PeerSecret, Expiration}).

%%%===================================================================
%%% gen_server callbacks
%%%===================================================================

-spec init(term()) -> {ok, state()}.
init(_) ->
    net_kernel:monitor_nodes(true),
    {ok, no_state}.

-spec handle_call(term(), {pid(), term()}, state()) ->
                         {reply, term(), state()}.
handle_call(_, _, State) ->
    {reply, not_implemented, State}.

-spec handle_info(term(), state()) -> {noreply, state()}.
handle_info({nodeup, _Node}, State) ->
    {noreply, State};
handle_info({nodedown, Node}, State) ->
    rtransfer_link_table:map_all_for_node(
      Node,
      fun(Rows) ->
              lists:map(
                fun(#{ref := Ref, request := Req0} = Row) ->
                        NewNode = random_node([Node]),
                        Req = Req0#{ref := Ref},
                        gen_server2:cast({rtransfer_link, NewNode}, {fetch, Req}),
                        Row#{node := NewNode}
                end,
                Rows)
      end),
    {noreply, State};
handle_info(Message, State) ->
    lager:debug("Ignoring message: ~p", [Message]),
    {noreply, State}.

-spec handle_cast(term(), state()) -> {noreply, state()}.
handle_cast(Message, State) ->
    lager:debug("Ignoring cast: ~p", [Message]),
    {noreply, State}.

-spec code_change(term(), state(), term()) -> {ok, state()}.
code_change(_, State, _Extra) ->
    {ok, State}.

-spec terminate(term(), state()) -> ok.
terminate(_, _) ->
    ok.

%%%===================================================================
%%% Helpers
%%%===================================================================

-spec random_node() -> node().
random_node() ->
    random_node([]).

-spec random_node(WithoutNodes :: [node()]) -> node().
random_node(WithoutNodes) ->
    Nodes = rtransfer_link_table:nodes() -- WithoutNodes,
    rtransfer_link_utils:random_element(Nodes).
