%%%-------------------------------------------------------------------
%%% @author Konrad Zemek
%%% @copyright (C) 2017 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% @end
%%%-------------------------------------------------------------------
-module(rtransfer_link_port).
-author("Konrad Zemek").

-behaviour(gen_server).

%%%===================================================================
%%% Type definitions
%%%===================================================================

-type request() :: map().
-type req_id() :: binary().
-type from() :: {pid(), Tag :: term()}.

-type state() :: #{
             port := port(),
             req_map := #{req_id() => from()},
             buffer := [binary()]
            }.

-export_type([req_id/0]).

%%%===================================================================
%%% Exports
%%%===================================================================

-export([start_link/0, request/1, answer/1, sync_request/1, sync_request/2]).
-export([init/1, handle_call/3, handle_cast/2, handle_info/2,
         code_change/3, terminate/2]).

%%%===================================================================
%%% API
%%%===================================================================

-spec start_link() -> {ok, pid()} | {error, any()}.
start_link() ->
    gen_server2:start_link({local, ?MODULE}, ?MODULE, {}, []).

-spec request(request()) -> req_id().
request(#{req_id := ReqId} = Request) ->
    JSONRequest = jiffy:encode(Request),
    gen_server2:cast(?MODULE, {request, self(), ReqId, JSONRequest}),
    ReqId;
request(Request) when is_map(Request) ->
    request(Request#{req_id => make_req_id()}).

-spec answer(request()) -> ok.
answer(#{req_id := _} = Answer) ->
    JSONAnswer = jiffy:encode(Answer#{is_answer => true}),
    gen_server2:cast(?MODULE, {answer, JSONAnswer}).

-spec sync_request(request()) -> {error, timeout} | term().
sync_request(Request) ->
    sync_request(Request, 60000).

-spec sync_request(request(), Timeout :: non_neg_integer()) ->
                          {error, timeout} | term().
sync_request(Request, Timeout) ->
    ReqId = request(Request),
    receive
        {response, ReqId, Response} -> Response
    after
        Timeout -> {error, timeout}
    end.

%%%===================================================================
%%% gen_server callbacks
%%%===================================================================

-spec init(term()) -> {ok, state()}.
init(_) ->
    ExecPath = get_exec_path(),
    Args = make_args(),
    Env = make_env(),
    lager:info("Starting port ~s ~s ~s", [[[E, "=", V] || {E, V} <- Env],
                                          ExecPath, lists:join(" ", Args)]),
    Port = erlang:open_port({spawn_executable, ExecPath},
                            [exit_status, {args, Args}, {env, Env},
                             {line, 1024}, binary, {parallelism, true}]),
    {ok, #{
       port => Port,
       req_map => #{},
       buffer => []}}.

-spec handle_call(term(), {pid(), term()}, state()) ->
                         {reply, term(), state()}.
handle_call(_Request, _From, State) ->
    {reply, not_implemented, State}.

-spec handle_cast(term(), state()) -> {noreply, state()}.
handle_cast({answer, Answer}, #{port := Port} = State) ->
    lager:debug("Sending link answer: ~s", [Answer]),
    true = port_command(Port, [Answer, $\n]),
    {noreply, State};

handle_cast({request, From, ReqId, Request},
            #{port := Port, req_map := ReqMap} = State) ->
    lager:debug("Sending link request: ~s", [Request]),
    true = port_command(Port, [Request, $\n]),
    NewReqMap = maps:put(ReqId, From, ReqMap),
    {noreply, State#{req_map := NewReqMap}}.

-spec handle_info(term(), state()) -> {noreply, state()}.
handle_info({end_req, ReqId}, #{req_map := ReqMap} = State) ->
    NewReqMap = maps:remove(ReqId, ReqMap),
    {noreply, State#{req_map := NewReqMap}};

handle_info({'EXIT', _Port, Reason}, State) ->
    lager:error("link port shut down: ~p", [Reason]),
    {stop, {shutdown, link_port_down}, State};

handle_info({_Port, {exit_status, Status}}, State) ->
    lager:error("link port exited with ~p", [Status]),
    {stop, {shutdown, link_port_down}, State};

handle_info({_Port, {data, {noeol, Data}}}, #{buffer := Buf} = State) ->
    {noreply, State#{buffer := [Data | Buf]}};

handle_info({_Port, {data, {eol, Data}}}, #{buffer := Buf} = State) ->
    NewState = handle_data(lists:reverse([Data | Buf]), State),
    {noreply, NewState}.

-spec code_change(term(), state(), term()) -> {ok, state()}.
code_change(_OldVsn, State, _Extra) ->
    {ok, State}.

-spec terminate(term(), state()) -> any().
terminate(_Reason, #{port := Port}) ->
    try erlang:port_close(Port)
    catch _:_ -> ok
    end.

%%%===================================================================
%%% Helpers
%%%===================================================================

-spec get_exec_path() -> string().
get_exec_path() ->
    rtransfer_link_utils:app_path(priv, ["link"]).

-spec handle_data(iodata(), state()) -> state().
handle_data(#{<<"isQuestion">> := true} = Question, State) ->
    rtransfer_link_questions:handle(Question),
    State;

handle_data(#{<<"disconnected">> := #{<<"connectionId">> := ConnId}}, State) ->
    rtransfer_link_connection:disconnected(ConnId),
    State;

handle_data(#{} = Response, #{req_map := ReqMap} = State) ->
    ReqId = maps:get(<<"reqId">>, Response, undefined),
    Recipient = maps:get(ReqId, ReqMap, rtransfer_link),
    Recipient ! {response, ReqId, error_response_to_tuple(Response)},
    case maps:get(<<"isUpdate">>, Response, false) of
        true -> State;
        false ->
            erlang:send_after(15000, self(), {end_req, ReqId}),
            State
    end;

handle_data(Data, State) ->
    lager:debug("Received link response: ~s", [Data]),
    Response = jiffy:decode(Data, [return_maps]),
    handle_data(Response, State).

-spec make_req_id() -> binary().
make_req_id() ->
    uuid:uuid_to_string(uuid:get_v4(), binary_nodash).

-spec error_response_to_tuple(term()) -> {error, term()} | term().
error_response_to_tuple(#{<<"error">> := #{<<"description">> := Reason}}) ->
    {error, Reason};
error_response_to_tuple(Response) ->
    Response.


-spec make_args() -> [string()].
make_args() ->
    Transfer = application:get_env(rtransfer_link, transfer, []),
    DescriptorCache = application:get_env(rtransfer_link, descriptor_cache, []),
    Monitoring = application:get_env(rtransfer_link, monitoring, []),
    SSL = application:get_env(rtransfer_link, ssl, []),
    Shaper = application:get_env(rtransfer_link, shaper, []),
    LogDir = application:get_env(rtransfer_link, logdir, "/var/log/op_worker"),
    Verbosity = application:get_env(rtransfer_link, verbosity, undefined),
    HelperWorkers = application:get_env(rtransfer_link, helper_workers, 100),
    WebDAVHelperWorkers = application:get_env(rtransfer_link, webdav_helper_workers, 25),

    ServerPort = proplists:get_value(server_port, Transfer, 6665),
    BlockSize = proplists:get_value(block_size, Transfer, 26240000),
    DataConnsPerLink = proplists:get_value(data_conns_per_link, Transfer, 10),
    DataConnRecvBufSize = proplists:get_value(recv_buffer_size, Transfer, 52428800),
    DataConnSendBufSize = proplists:get_value(send_buffer_size, Transfer, 52428800),
    MaxIncomingBufferedSize = proplists:get_value(max_incoming_buffered_size, Transfer, 20971520),
    StorageBuckets = proplists:get_value(storage_buckets, Transfer, 100),
    CongestionFlavor = proplists:get_value(send_congestion_flavor, Transfer, undefined),
    ThroughputProbeInterval = proplists:get_value(throughput_probe_interval, Transfer, 3000),

    MaxOpenDescriptors =
        proplists:get_value(max_open_descriptors, DescriptorCache, 1000),
    DescriptorCacheDuration =
        proplists:get_value(descriptor_cache_duration, DescriptorCache, 60000),
    DescriptorCacheTick =
        proplists:get_value(descriptor_cache_tick, DescriptorCache, 1000),

    GraphiteUrl = proplists:get_value(graphite_url, Monitoring),
    GNamespacePrefix = proplists:get_value(graphite_namespace_prefix, Monitoring, "rtransfer_link"),
    GReportingPeriod = proplists:get_value(graphite_reporting_period, Monitoring, 60),
    GReportingFull = proplists:get_value(graphite_reporting_full, Monitoring, false),

    UseSSL = proplists:get_value(use_ssl, SSL),
    SSLCertPath = proplists:get_value(cert_path, SSL),
    SSLKeyPath = proplists:get_value(key_path, SSL),
    SSLPasswordPath = proplists:get_value(password_path, SSL),
    SSLCAPath = proplists:get_value(ca_path, SSL),
    ECCCurveName = proplists:get_value(ecc_curve_name, SSL),
    Ciphers = proplists:get_value(ciphers, SSL),

    QuantumMsSize = proplists:get_value(quantum_ms_size, Shaper, 25),

    SubLogDir = filename:join(LogDir, "link"),
    case filelib:ensure_dir(filename:join(SubLogDir, "dummy.file")) of
        ok -> ok;
        Error ->
            lager:error("Cannot create log directory ~s for rtransfer_link: ~p",
                        [SubLogDir, Error])
    end,

    Opts = [
            {v, Verbosity},
            {log_dir, SubLogDir},
            {log_link, LogDir},
            {server_port, ServerPort},
            {single_fetch_max_size, BlockSize},
            {graphite_url, GraphiteUrl},
            {graphite_namespace_prefix, GNamespacePrefix},
            {graphite_reporting_period, GReportingPeriod},
            {graphite_reporting_full, GReportingFull},
            {max_open_descriptors, MaxOpenDescriptors},
            {descriptor_cache_duration, DescriptorCacheDuration},
            {descriptor_cache_tick, DescriptorCacheTick},
            {number_of_data_conns, DataConnsPerLink},
            {recv_buf_size, DataConnRecvBufSize},
            {send_buf_size, DataConnSendBufSize},
            {use_ssl, UseSSL},
            {ssl_cert_path, SSLCertPath},
            {ssl_key_path, SSLKeyPath},
            {ssl_password_path, SSLPasswordPath},
            {ssl_ca_path, SSLCAPath},
            {ssl_ecc_curve_name, ECCCurveName},
            {ssl_cipher_list, Ciphers},
            {max_incoming_buffered_size, MaxIncomingBufferedSize},
            {storage_buckets, StorageBuckets},
            {send_congestion_flavor, CongestionFlavor},
            {shaper_quantum_ms_size, QuantumMsSize},
            {helper_workers, HelperWorkers},
            {webdav_helper_workers, WebDAVHelperWorkers},
            {throughput_probe_interval, ThroughputProbeInterval}
           ],

    lists:filtermap(
      fun
          ({_Opt, false}) -> false;
          ({_Opt, undefined}) -> false;
          ({Opt, true}) -> {true, "--" ++ atom_to_list(Opt)};
          ({Opt, Value}) -> {true, "--" ++ atom_to_list(Opt) ++ "=" ++ to_list(Value)}
      end,
      Opts).

make_env() ->
    CPUProfilingPath = application:get_env(rtransfer_link, cpu_profiling_path, undefined),
    case CPUProfilingPath of
        Path when is_list(Path) -> [{"CPUPROFILE", CPUProfilingPath}];
        undefined -> []
    end.

to_list(Int) when is_integer(Int) -> integer_to_list(Int);
to_list(Atom) when is_atom(Atom) -> atom_to_list(Atom);
to_list(List) when is_list(List) -> List.
