%%%-------------------------------------------------------------------
%%% @author Lukasz Opiola
%%% @copyright (C) 2018 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc
%%% Volatile authorization nonce record that holds single-use random nonces
%%% used for establishing safe inter-provider communication.
%%% @end
%%%-------------------------------------------------------------------
-module(authorization_nonce).
-author("Lukasz Opiola").

-include("modules/datastore/datastore_models.hrl").
-include_lib("global_definitions.hrl").
-include_lib("ctool/include/logging.hrl").

%% API
-export([create/0, verify/1, delete/1]).

%% datastore_model callbacks
-export([get_ctx/0]).

-type nonce() :: binary().

-define(NONCE_TTL, application:get_env(
    ?APP_NAME, provider_authorization_nonce_ttl_sec, 30
)).

-define(CTX, #{
    model => ?MODULE,
    disc_driver => undefined
}).

%%%===================================================================
%%% API
%%%===================================================================

%%--------------------------------------------------------------------
%% @doc
%% Creates a new, random provider nonce.
%% @end
%%--------------------------------------------------------------------
-spec create() -> {ok, nonce()} | {error, term()}.
create() ->
    {ok, #document{key = Nonce}} = datastore_model:save(?CTX, #document{
        value = #authorization_nonce{
            timestamp = time_utils:cluster_time_seconds()
        }
    }),
    {ok, Nonce}.

%%--------------------------------------------------------------------
%% @doc
%% Verifies if given nonce was generated by this provider and consumes it
%% immediately. Expired nonces are considered invalid.
%% @end
%%--------------------------------------------------------------------
-spec verify(nonce()) -> boolean().
verify(Nonce) ->
    case datastore_model:get(?CTX, Nonce) of
        {ok, #document{value = #authorization_nonce{timestamp = Timestamp}}} ->
            ok = delete(Nonce),
            Timestamp + ?NONCE_TTL >= time_utils:cluster_time_seconds();
        _ ->
            false
    end.

%%--------------------------------------------------------------------
%% @doc
%% Deletes given nonce.
%% @end
%%--------------------------------------------------------------------
-spec delete(nonce()) -> ok | {error, term()}.
delete(Nonce) ->
    datastore_model:delete(?CTX, Nonce).

%%%===================================================================
%%% datastore_model callbacks
%%%===================================================================

%%--------------------------------------------------------------------
%% @doc
%% Returns model's context.
%% @end
%%--------------------------------------------------------------------
-spec get_ctx() -> datastore:ctx().
get_ctx() ->
    ?CTX.