%%%--------------------------------------------------------------------
%%% @author Michal Zmuda
%%% @copyright (C) 2016 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%--------------------------------------------------------------------
%%% @doc This module is responsible for subscriptions WSS listener starting
%%% and stopping.
%%% @end
%%%--------------------------------------------------------------------
-module(subscriptions_wss_listener).
-author("Michal Zmuda").

-include("registered_names.hrl").
-include_lib("ctool/include/logging.hrl").

-behaviour(listener_behaviour).

%% listener_behaviour callbacks
-export([port/0, start/0, stop/0, healthcheck/0]).

-define(WSS_LISTENER, subscriptions_wss).

%%%===================================================================
%%% listener_behaviour callbacks
%%%===================================================================

%%--------------------------------------------------------------------
%% @doc
%% {@link listener_behaviour} callback port/0.
%% @end
%%--------------------------------------------------------------------
-spec port() -> integer().
port() ->
    {ok, WssPort} = application:get_env(?APP_NAME, subscriptions_wss_port),
    WssPort.


%%--------------------------------------------------------------------
%% @doc
%% {@link listener_behaviour} callback start/1.
%% @end
%%--------------------------------------------------------------------
-spec start() -> ok | {error, Reason :: term()}.
start() ->
    try
        % Get rest config
        Port = port(),
        {ok, HttpsAcceptors} = application:get_env(?APP_NAME, subscroptions_https_acceptors),

        % Get certs
        {ok, ZoneCADir} = application:get_env(?APP_NAME, ozpca_dir),
        {ok, ZoneCaCertPem} = file:read_file(ozpca:cacert_path(ZoneCADir)),

        {ok, KeyFile} = application:get_env(?APP_NAME, web_key_file),
        {ok, CertFile} = application:get_env(?APP_NAME, web_cert_file),
        {ok, CaCertsDir} = application:get_env(?APP_NAME, cacerts_dir),
        {ok, CaCertPems} = file_utils:read_files({dir, CaCertsDir}),
        CaCerts = lists:map(fun cert_decoder:pem_to_der/1, [ZoneCaCertPem | CaCertPems]),
        Dispatch = [
            {'_', [
                {"/subscriptions/[...]", subscriptions_wss_handler, []}
            ]}
        ],

        {ok, _} = ranch:start_listener(?WSS_LISTENER, HttpsAcceptors,
            ranch_ssl, [
                {port, Port},
                % @todo Use gui cert files rather than certs generated by GR, since
                % we don't yet have a mechanism of distributing the CA cert.
                {keyfile, KeyFile},
                {certfile, CertFile},
                {cacerts, CaCerts},
                {verify, verify_peer},
                {ciphers, ssl_utils:safe_ciphers()}
            ], cowboy_protocol,
            [
                {env, [{dispatch, cowboy_router:compile(Dispatch)}]}
            ]),
        ok
    catch
        _Type:Error ->
            ?error_stacktrace("Could not start wss, error: ~p", [Error]),
            {error, Error}
    end.


%%--------------------------------------------------------------------
%% @doc
%% {@link listener_behaviour} callback stop/0.
%% @end
%%--------------------------------------------------------------------
-spec stop() -> ok | {error, Reason :: term()}.
stop() ->
    case catch ranch:stop_listener(?WSS_LISTENER) of
        (ok) ->
            ok;
        (Error) ->
            ?error("Error on stopping listener ~p: ~p", [?WSS_LISTENER, Error]),
            {error, subscriptions_wss_stop_error}
    end.


%%--------------------------------------------------------------------
%% @doc
%% {@link listener_behaviour} callback healtcheck/0.
%% @end
%%--------------------------------------------------------------------
-spec healthcheck() -> ok | {error, server_not_responding}.
healthcheck() ->
    Endpoint = "https://127.0.0.1:" ++ integer_to_list(port()),
    case http_client:get(Endpoint, #{}, <<>>, [insecure]) of
        {ok, _, _, _} -> ok;
        _ -> {error, server_not_responding}
    end.