%%%-------------------------------------------------------------------
%%% @author Konrad Zemek
%%% @copyright (C): 2014 ACK CYFRONET AGH
%%% This software is released under the MIT license
%%% cited in 'LICENSE.txt'.
%%% @end
%%%-------------------------------------------------------------------
%%% @doc The module implementing the business logic for OpenID Connect end-user
%%% authentication and authorization.
%%% @end
%%%-------------------------------------------------------------------
-module(auth_logic).
-author("Konrad Zemek").

-include("auth_common.hrl").
-include("gui/common.hrl").
-include("registered_names.hrl").
-include_lib("ctool/include/logging.hrl").
-include("datastore/oz_datastore_models_def.hrl").
-include("datastore/oz_datastore_models_def.hrl").
-include_lib("hackney/include/hackney_lib.hrl").

% String that will be placed in macaroons' location field
-define(MACAROONS_LOCATION, <<"onezone">>).

%% API
-export([get_redirection_uri/2]).
-export([gen_token/1, gen_token/2, validate_token/5,
    invalidate_token/1, invalidate_user_tokens/1,
    authenticate_user/1]).

%% Handling state tokens
-export([generate_state_token/2, lookup_state_token/1]).

%%%===================================================================
%%% API
%%%===================================================================

%%--------------------------------------------------------------------
%% @doc
%% Attempt to authenticate user versus user id found in database.
%% @end
%%--------------------------------------------------------------------
-spec authenticate_user(Identifier :: binary()) -> {ok, Token :: binary()} | {error, any()}.
authenticate_user(Identifier) ->
    % @todo really authenticate; we're just pretending
    case onedata_auth:get(Identifier) of
        {ok, #document{value = #onedata_auth{secret = Secret, user_id = UserId}}} ->
            % @todo yeah, that seems like a very authenticated UserId
            UserId = UserId,

            {ok, ExpirationSecs} = application:get_env(?APP_NAME,
                authentication_macaroon_expiration_seconds),

            Location = ?MACAROONS_LOCATION,
            M = macaroon:create(Location, Secret, Identifier),
            M2 = macaroon:add_first_party_caveat(M,
                ["time < ", integer_to_binary(time_utils:cluster_time_seconds() + ExpirationSecs)]),

            case token_utils:serialize62(M2) of
                {ok, _} = Serialized -> Serialized;
                {error, _} = Error -> Error
            end;

        _ ->
            {error, no_auth_record}
    end.


%%--------------------------------------------------------------------
%% @doc Returns provider hostname and a full URI to which the user should be
%% redirected from the onezone. The redirection is part of the OpenID
%% flow and the URI contains an Authorization token. The provider hostname
%% is useful to check connectivity before redirecting.
%% @end
%%--------------------------------------------------------------------
-spec get_redirection_uri(UserId :: binary(), ProviderId :: binary()) ->
    {ok, RedirectionUri :: binary()}.
get_redirection_uri(UserId, ProviderId) ->
    Token = gen_token(UserId, ProviderId),
    {ok, _} = od_user:update(UserId, #{chosen_provider => ProviderId}),
    {ok, ProviderURL} = provider_logic:get_url(ProviderId),
    URL = str_utils:format_bin("~s~s?code=~s", [
        ProviderURL, ?provider_auth_endpoint, Token
    ]),
    {ok, URL}.


%%--------------------------------------------------------------------
%% @doc Creates an authorization code for a native client.
%%--------------------------------------------------------------------
-spec gen_token(UserId :: binary()) -> Token :: binary().
gen_token(UserId) ->
    Secret = generate_secret(),
    Caveats = [],%["method = GET", "rootResource in spaces,user"],
    {ok, IdentifierBinary} = onedata_auth:save(#document{value = #onedata_auth{
        secret = Secret, user_id = UserId}}),
    Identifier = binary_to_list(IdentifierBinary),
    M = create_macaroon(Secret, str_utils:to_binary(Identifier), Caveats),
    {ok, Token} = token_utils:serialize62(M),
    Token.

%%--------------------------------------------------------------------
%% @doc Creates an authorization code for a Provider.
%%--------------------------------------------------------------------
-spec gen_token(UserId :: binary(), ProviderId :: binary() | undefined) ->
    Token :: binary().
gen_token(UserId, _ProviderId) ->
    Secret = generate_secret(),
    Location = ?MACAROONS_LOCATION,
    {ok, IdentifierBinary} = onedata_auth:save(#document{value = #onedata_auth{
        secret = Secret, user_id = UserId}}),
    Identifier = binary_to_list(IdentifierBinary),
    %% @todo: VFS-1869
    M = create_macaroon(Secret, str_utils:to_binary(Identifier), []),

    CaveatKey = generate_secret(),
    {ok, CaveatIdBinary} = onedata_auth:save(#document{value = #onedata_auth{
        secret = CaveatKey, user_id = UserId}}),
    CaveatId = binary_to_list(CaveatIdBinary),
    M2 = macaroon:add_third_party_caveat(M, Location, CaveatKey, CaveatId),
    {ok, Token} = token_utils:serialize62(M2),
    Token.

%%--------------------------------------------------------------------
%% @doc Validates an access token for an OpenID client and returns a UserId of
%% the user that gave the authorization.
%% @end
%%--------------------------------------------------------------------
-spec validate_token(ProviderId :: binary(), Macaroon :: macaroon:macaroon(),
    DischargeMacaroons :: [macaroon:macaroon()], Method :: binary(),
    RootResource :: atom()) ->
    {ok, UserId :: binary()} | {error, Reason :: any()}.
validate_token(ProviderId, Macaroon, DischargeMacaroons, _Method, _RootResource) ->
    Identifier = macaroon:identifier(Macaroon),
    case onedata_auth:get(Identifier) of
        {ok, #document{value = #onedata_auth{secret = Secret, user_id = UserId}}} ->
            V = macaroon_verifier:create(),

            VerifyFun = fun
                (<<"time < ", Integer/binary>>) ->
                    time_utils:cluster_time_seconds() < binary_to_integer(Integer);
                (<<"providerId = ", PID/binary>>) ->
                    PID =:= ProviderId;
                (_) ->
                    false
            end,

            V1 = macaroon_verifier:satisfy_general(V, VerifyFun),
            case macaroon_verifier:verify(V1, Macaroon, Secret, DischargeMacaroons) of
                ok -> {ok, UserId};
                {error, Reason} -> {error, Reason}
            end;

        _ ->
            {error, unknown_macaroon}
    end.


%%--------------------------------------------------------------------
%% @doc Invalidates all auth tokens of given user.
%% @end
%%--------------------------------------------------------------------
-spec invalidate_user_tokens(UserId :: od_user:id()) -> ok.
invalidate_user_tokens(UserId) ->
    {ok, AuthDocs} = onedata_auth:get_auth_by_user_id(UserId),
    lists:foreach(
        fun(#document{key = AuthId}) ->
            invalidate_token(AuthId)
        end, AuthDocs).


%%--------------------------------------------------------------------
%% @doc Invalidates a given auth token.
%% @end
%%--------------------------------------------------------------------
-spec invalidate_token(binary()) -> ok.
invalidate_token(Identifier) when is_binary(Identifier) ->
    onedata_auth:delete(Identifier).


%%--------------------------------------------------------------------
%% @doc Generates a state token and returns it. In the process, it stores the token
%% and associates some login info, that can be later retrieved given the token.
%% For example, where to redirect the user after login.
%% @end
%%--------------------------------------------------------------------
-spec generate_state_token(HandlerModule :: atom(), ConnectAccount :: boolean()) ->
    state_token:id().
generate_state_token(HandlerModule, ConnectAccount) ->
    StateInfo = #{
        module => HandlerModule,
        connect_account => ConnectAccount,
        % Right now this always redirects to main page, although
        % might be used in the future.
        redirect_after_login => <<?PAGE_AFTER_LOGIN>>
    },
    {ok, Token} = state_token:create_state_token(StateInfo),
    Token.

%%--------------------------------------------------------------------
%% @doc Checks if the given state token exists and returns login info
%% associated with it or error otherwise.
%% @end
%%--------------------------------------------------------------------
-spec lookup_state_token(Token :: binary()) ->
    {ok, state_token:state_info()} | error.
lookup_state_token(Token) ->
    state_token:lookup_state_token(Token).

%%%===================================================================
%%% Internal functions
%%%===================================================================

%%--------------------------------------------------------------------
%% @private
%% @doc
%% Creates a macaroon with expiration time read from
%% `authorization_macaroon_expiration_seconds` environment variable.
%% @end
%%--------------------------------------------------------------------
-spec create_macaroon(Secret :: iodata(), Identifier :: iodata(),
    Caveats :: [iodata()]) -> macaroon:macaroon().
create_macaroon(Secret, Identifier, Caveats) ->
    {ok, ExpirationSeconds} = application:get_env(?APP_NAME,
        authorization_macaroon_expiration_seconds),
    ExpirationTime = time_utils:cluster_time_seconds() + ExpirationSeconds,

    Location = ?MACAROONS_LOCATION,

    lists:foldl(
        fun(Caveat, Macaroon) ->
            macaroon:add_first_party_caveat(Macaroon, Caveat)
        end,
        macaroon:create(Location, Secret, Identifier),
        [["time < ", integer_to_binary(ExpirationTime)] | Caveats]).

%%--------------------------------------------------------------------
%% @private
%% @doc
%% Generates a hex-encoded random secret for use with Macaroon.
%% @end
%%--------------------------------------------------------------------
-spec generate_secret() -> binary().
generate_secret() ->
    BinSecret = crypto:strong_rand_bytes(macaroon:suggested_secret_length()),
    <<<<Y>> || <<X:4>> <= BinSecret, Y <- integer_to_list(X, 16)>>.
